<?php

declare (strict_types=1);

require_once __DIR__ . '/../../support/supportpal/Bootstrap/autoload.php';
/**
 * SupportPal Integration Addon
 *
 * Addon which allows configuration for SupportPal-WHMCS integration.
 */
use SupportPal\WhmcsIntegration\DI\Container;
use SupportPal\WhmcsIntegration\Exception\ValidationException;
use SupportPal\WhmcsIntegration\Helper\CsrfTokenHelper;
use SupportPal\WhmcsIntegration\Helper\SessionHelper;
use SupportPal\WhmcsIntegration\Repository\Whmcs\ModSupportPalRepository;
use SupportPal\WhmcsIntegration\Validator\ArticleTypeIdBelongsToBrandValidator;
use SupportPal\WhmcsIntegration\Validator\AuthenticationValidator;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Request;
use WHMCS\Database\Capsule;
/**
 * Configuration options for add-on.
 *
 * @return mixed[]
 */
function supportpal_config() : array
{
    return ['name' => 'SupportPal Integration', 'description' => 'Provides configuration for the SupportPal support module, which replaces the support components of WHMCS with those of SupportPal.', 'version' => '4.7.0', 'language' => 'english', 'author' => 'SupportPal'];
}
/**
 * Activate addon
 *
 * @return mixed[]
 */
function supportpal_activate() : array
{
    try {
        // Drop module table if it exists somehow.
        \WHMCS\Database\Capsule::schema()->dropIfExists('mod_supportpal');
        // Create module table.
        \WHMCS\Database\Capsule::schema()->create('mod_supportpal', function ($table) {
            $table->increments('id');
            $table->string('name');
            $table->string('value', 255);
            $table->timestamps();
        });
        // Add integration configuration rows.
        \WHMCS\Database\Capsule::table('mod_supportpal')->insert([['name' => 'base_url', 'value' => ''], ['name' => 'admin_url', 'value' => ''], ['name' => 'api_token', 'value' => ''], ['name' => 'brand_id', 'value' => 1], ['name' => 'related_service_field', 'value' => ''], ['name' => 'article_suggestions', 'value' => 1], ['name' => 'announcements_id', 'value' => ''], ['name' => 'knowledgebase_id', 'value' => ''], ['name' => 'downloads_id', 'value' => '']]);
        // Update support module in configuration.
        \WHMCS\Database\Capsule::table('tblconfiguration')->where('setting', 'LIKE', 'SupportModule')->update(['value' => 'supportpal']);
        return ['status' => 'success', 'description' => 'The SupportPal integration is now active and can be configured, please make sure your admin role group can access the module and then configure the integration by going to Addons -> SupportPal Integration.'];
    } catch (\Exception $e) {
        // Something went wrong
        return ['status' => 'error', 'description' => 'The SupportPal integration could not be activated.'];
    }
}
/**
 * Deactivate addon
 *
 * @return mixed[]
 */
function supportpal_deactivate() : array
{
    try {
        // Drop module table.
        \WHMCS\Database\Capsule::schema()->dropIfExists('mod_supportpal');
        // Update support module in configuration.
        \WHMCS\Database\Capsule::table('tblconfiguration')->where('setting', 'LIKE', 'SupportModule')->update(['value' => '']);
        return ['status' => 'success', 'description' => 'The SupportPal integration has been deactivated.'];
    } catch (\Exception $e) {
        // Something went wrong
        return ['status' => 'error', 'description' => 'The SupportPal integration could not be deactivated.'];
    }
}
/**
 * Upgrade addon
 *
 * @param mixed[] $vars
 */
function supportpal_upgrade(array $vars) : void
{
    // Version 3.2.0 and above
    if (\version_compare($vars['version'], '3.2') === -1) {
        // Create module table.
        \WHMCS\Database\Capsule::schema()->create('mod_supportpal', function ($table) {
            $table->increments('id');
            $table->string('name');
            $table->string('value', 255);
            $table->timestamps();
        });
        // Add integration configuration rows.
        \WHMCS\Database\Capsule::table('mod_supportpal')->insert([['name' => 'base_url', 'value' => $vars['base_url']], ['name' => 'api_token', 'value' => $vars['api_token']], ['name' => 'brand_id', 'value' => 1], ['name' => 'related_service_field', 'value' => ''], ['name' => 'article_suggestions', 'value' => 1], ['name' => 'announcements_id', 'value' => $vars['announcements_id']], ['name' => 'knowledgebase_id', 'value' => $vars['knowledgebase_id']], ['name' => 'downloads_id', 'value' => $vars['downloads_id']]]);
    }
    // Version 3.4.0 and above
    if (\version_compare($vars['version'], '3.4.0') === -1) {
        /** @var ModSupportPalRepository $repository */
        $repository = \SupportPal\WhmcsIntegration\DI\Container::getDefaultContainer()->get(\SupportPal\WhmcsIntegration\Repository\Whmcs\ModSupportPalRepository::class);
        $baseUrl = $repository->getDefaultBaseUrl();
        // Add integration configuration row.
        \WHMCS\Database\Capsule::table('mod_supportpal')->insert([['name' => 'admin_url', 'value' => $baseUrl !== null ? \rtrim($baseUrl, '/') . '/admin/' : '']]);
    }
    if (\version_compare($vars['version'], '4.0.0') !== -1) {
        return;
    }
    // Version 4.0.0 and above
    // Remove integration configuration row.
    \WHMCS\Database\Capsule::table('mod_supportpal')->where('name', 'simpleauth_key')->delete();
    // Add integration configuration rows.
    \WHMCS\Database\Capsule::table('mod_supportpal')->insert([['name' => 'announcements_enabled', 'value' => 1], ['name' => 'knowledgebase_enabled', 'value' => 1], ['name' => 'downloads_enabled', 'value' => 1]]);
}
/**
 * Addon Page
 *
 * @param mixed[] $vars
 * @throws Exception
 */
function supportpal_output(array $vars) : void
{
    // Create Smarty object.
    $smarty = new \Smarty();
    // Add language file.
    $smarty->assign('LANG', $vars['_lang']);
    $request = \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Request::createFromGlobals();
    // Do we have a POST request to update the configuration?
    if (!empty($request->request->all())) {
        try {
            \WHMCS\Database\Capsule::table('mod_supportpal')->updateOrInsert(['name' => 'base_url'], ['value' => $request->get('base_url', '')]);
            \WHMCS\Database\Capsule::table('mod_supportpal')->updateOrInsert(['name' => 'admin_url'], ['value' => $request->get('admin_url', '')]);
            \WHMCS\Database\Capsule::table('mod_supportpal')->updateOrInsert(['name' => 'api_token'], ['value' => $request->get('api_token', '')]);
            \WHMCS\Database\Capsule::table('mod_supportpal')->updateOrInsert(['name' => 'brand_id'], ['value' => $request->get('brand_id', '')]);
            \WHMCS\Database\Capsule::table('mod_supportpal')->updateOrInsert(['name' => 'related_service_field'], ['value' => $request->get('related_service_field', '')]);
            \WHMCS\Database\Capsule::table('mod_supportpal')->updateOrInsert(['name' => 'article_suggestions'], ['value' => $request->get('article_suggestions') !== null ? 1 : 0]);
            \WHMCS\Database\Capsule::table('mod_supportpal')->updateOrInsert(['name' => 'announcements_enabled'], ['value' => $request->get('announcements_enabled', '')]);
            \WHMCS\Database\Capsule::table('mod_supportpal')->updateOrInsert(['name' => 'announcements_id'], ['value' => $request->get('announcements_id', '')]);
            \WHMCS\Database\Capsule::table('mod_supportpal')->updateOrInsert(['name' => 'knowledgebase_enabled'], ['value' => $request->get('knowledgebase_enabled', '')]);
            \WHMCS\Database\Capsule::table('mod_supportpal')->updateOrInsert(['name' => 'knowledgebase_id'], ['value' => $request->get('knowledgebase_id', '')]);
            \WHMCS\Database\Capsule::table('mod_supportpal')->updateOrInsert(['name' => 'downloads_enabled'], ['value' => $request->get('downloads_enabled', '')]);
            \WHMCS\Database\Capsule::table('mod_supportpal')->updateOrInsert(['name' => 'downloads_id'], ['value' => $request->get('downloads_id', '')]);
            // All done, show success message
            $smarty->assign('success_message', $vars['_lang']['config_updated']);
        } catch (\Exception $e) {
            // Something went wrong, show error message
            $smarty->assign('error_message', $vars['_lang']['config_not_updated']);
        }
    }
    // Fetch settings.
    $settings = [];
    foreach (\WHMCS\Database\Capsule::table('mod_supportpal')->get() as $setting) {
        $settings[$setting->name] = $setting->value;
    }
    // Assign variables.
    $smarty->assign('supportpal', $settings);
    // Set CSRF token.
    $smarty->assign('validate_token', (new \SupportPal\WhmcsIntegration\Helper\CsrfTokenHelper(new \SupportPal\WhmcsIntegration\Helper\SessionHelper()))->generateToken());
    // Check current integration status.
    if (!empty($settings['base_url']) && !empty($settings['api_token']) && !empty($settings['brand_id'])) {
        $container = \SupportPal\WhmcsIntegration\DI\Container::getDefaultContainer();
        /** @var AuthenticationValidator $authenticationValidator */
        $authenticationValidator = $container->get(\SupportPal\WhmcsIntegration\Validator\AuthenticationValidator::class);
        try {
            $authenticationValidator->validate((int) $settings['brand_id']);
            $announcementsId = $settings['announcements_enabled'] ? (int) $settings['announcements_id'] : 0;
            $knowledgebaseId = $settings['knowledgebase_enabled'] ? (int) $settings['knowledgebase_id'] : 0;
            $downloadsId = $settings['downloads_enabled'] ? (int) $settings['downloads_id'] : 0;
            if ($announcementsId !== 0 && $knowledgebaseId !== 0 && $downloadsId !== 0) {
                /** @var ArticleTypeIdBelongsToBrandValidator $typeIdValidator */
                $typeIdValidator = $container->get(\SupportPal\WhmcsIntegration\Validator\ArticleTypeIdBelongsToBrandValidator::class);
                $typeIds = \array_values([$announcementsId, $knowledgebaseId, $downloadsId]);
                $typeIdValidator->validateMany($typeIds, (int) $settings['brand_id']);
            }
            $smarty->assign('integration_status', 0);
        } catch (\SupportPal\WhmcsIntegration\Exception\ValidationException $validationException) {
            $smarty->assign('integration_status', 1);
            $smarty->assign('integration_error', $validationException->getMessage());
        }
    } else {
        // Not configured.
        $smarty->assign('integration_status', 2);
    }
    // Smarty settings.
    $smarty->setCaching(\Smarty::CACHING_OFF);
    $smarty->setCompileDir($GLOBALS['templates_compiledir']);
    // Display template from folder.
    $smarty->display(\dirname(__FILE__) . '/templates/settings.tpl');
}
