<?php

declare (strict_types=1);
namespace SupportPal\WhmcsIntegration\Controller\Ticket;

use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\InvalidArgumentException;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Ticket\ChannelSettings;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Ticket\Ticket;
use SupportPal\WhmcsIntegration\Exception\InvalidApiResponse;
use SupportPal\WhmcsIntegration\Exception\InvalidCaptchaException;
use SupportPal\WhmcsIntegration\Factory\ClientAreaFactory;
use SupportPal\WhmcsIntegration\Factory\ResponseFactory;
use SupportPal\WhmcsIntegration\Helper\AuthenticatedUserHelper;
use SupportPal\WhmcsIntegration\Helper\CaptchaHelper;
use SupportPal\WhmcsIntegration\Helper\CustomFieldsHelper;
use SupportPal\WhmcsIntegration\Helper\DateTimeFormatter;
use SupportPal\WhmcsIntegration\Helper\StringHelper;
use SupportPal\WhmcsIntegration\Repository\Core\CoreSettingsRepository;
use SupportPal\WhmcsIntegration\Repository\Ticket\ChannelSettingsRepository;
use SupportPal\WhmcsIntegration\Repository\Ticket\CustomFieldRepository;
use SupportPal\WhmcsIntegration\Repository\Ticket\DepartmentRepository;
use SupportPal\WhmcsIntegration\Repository\Ticket\PriorityRepository;
use SupportPal\WhmcsIntegration\Repository\Ticket\TicketSettingsRepository;
use SupportPal\WhmcsIntegration\Repository\UserRepository;
use SupportPal\WhmcsIntegration\Repository\Whmcs\ModSupportPalRepository;
use SupportPal\WhmcsIntegration\Request\Request;
use SupportPal\WhmcsIntegration\Service\Converter\ModelToArrayConverter;
use SupportPal\WhmcsIntegration\Service\Manager\Locale\LocaleManager;
use SupportPal\WhmcsIntegration\Service\Manager\Ticket\TicketManager;
use WHMCS\Application\Support\Facades\Lang;
use WHMCS\ClientArea;
use WHMCS\Input\Sanitize;
use WHMCS\User\User;
use function array_filter;
use function array_values;
use function explode;
use function filter_var;
use function is_numeric;
use function json_encode;
use function nl2br;
use function rtrim;
use function sha1;
use function sprintf;
use function strtolower;
use const FILTER_VALIDATE_EMAIL;
class SubmitTicketController extends \SupportPal\WhmcsIntegration\Controller\Ticket\BaseTicketController
{
    protected const DEFAULT_PAGE_TITLE = 'supportticketssubmitticket';
    protected const TYPE = 'submitticket';
    /** @var string[] */
    private $locale;
    /** @var DepartmentRepository */
    private $departmentRepository;
    /** @var ModelToArrayConverter */
    private $modelToArrayConverter;
    /** @var ModSupportPalRepository */
    private $modSupportPalRepository;
    /** @var CoreSettingsRepository */
    private $coreSettingsRepository;
    /** @var TicketSettingsRepository */
    private $ticketSettingsRepository;
    /** @var StringHelper */
    private $stringHelper;
    /** @var ChannelSettingsRepository */
    private $channelSettingsRepository;
    /** @var CustomFieldRepository */
    private $customFieldRepository;
    /** @var CaptchaHelper */
    private $captchaHelper;
    /** @var UserRepository */
    private $userRepository;
    /** @var AuthenticatedUserHelper */
    private $authenticatedUserHelper;
    /** @var TicketManager */
    private $ticketManager;
    /** @var PriorityRepository */
    private $priorityRepository;
    /** @var CustomFieldsHelper */
    private $customFieldsHelper;
    /** @var ResponseFactory */
    private $responseFactory;
    /** @var ClientAreaFactory */
    private $clientAreaFactory;
    public function __construct(\SupportPal\WhmcsIntegration\Helper\DateTimeFormatter $dateTimeFormatter, \SupportPal\WhmcsIntegration\Service\Manager\Locale\LocaleManager $localeManager, \SupportPal\WhmcsIntegration\Repository\Ticket\DepartmentRepository $departmentRepository, \SupportPal\WhmcsIntegration\Service\Converter\ModelToArrayConverter $modelToArrayConverter, \SupportPal\WhmcsIntegration\Repository\Whmcs\ModSupportPalRepository $modSupportPalRepository, \SupportPal\WhmcsIntegration\Repository\Core\CoreSettingsRepository $coreSettingsRepository, \SupportPal\WhmcsIntegration\Repository\Ticket\TicketSettingsRepository $ticketSettingsRepository, \SupportPal\WhmcsIntegration\Helper\StringHelper $stringHelper, \SupportPal\WhmcsIntegration\Repository\Ticket\ChannelSettingsRepository $channelSettingsRepository, \SupportPal\WhmcsIntegration\Repository\Ticket\CustomFieldRepository $customFieldRepository, \SupportPal\WhmcsIntegration\Helper\CaptchaHelper $captchaHelper, \SupportPal\WhmcsIntegration\Repository\UserRepository $userRepository, \SupportPal\WhmcsIntegration\Helper\AuthenticatedUserHelper $authenticatedUserHelper, \SupportPal\WhmcsIntegration\Service\Manager\Ticket\TicketManager $ticketManager, \SupportPal\WhmcsIntegration\Repository\Ticket\PriorityRepository $priorityRepository, \SupportPal\WhmcsIntegration\Helper\CustomFieldsHelper $customFieldsHelper, \SupportPal\WhmcsIntegration\Factory\ResponseFactory $responseFactory, \SupportPal\WhmcsIntegration\Factory\ClientAreaFactory $clientAreaFactory)
    {
        parent::__construct($dateTimeFormatter);
        $this->locale = $localeManager->getLocale(\WHMCS\Application\Support\Facades\Lang::trans('locale'));
        $this->departmentRepository = $departmentRepository;
        $this->modelToArrayConverter = $modelToArrayConverter;
        $this->modSupportPalRepository = $modSupportPalRepository;
        $this->coreSettingsRepository = $coreSettingsRepository;
        $this->ticketSettingsRepository = $ticketSettingsRepository;
        $this->stringHelper = $stringHelper;
        $this->channelSettingsRepository = $channelSettingsRepository;
        $this->customFieldRepository = $customFieldRepository;
        $this->captchaHelper = $captchaHelper;
        $this->userRepository = $userRepository;
        $this->authenticatedUserHelper = $authenticatedUserHelper;
        $this->ticketManager = $ticketManager;
        $this->priorityRepository = $priorityRepository;
        $this->customFieldsHelper = $customFieldsHelper;
        $this->responseFactory = $responseFactory;
        $this->clientAreaFactory = $clientAreaFactory;
    }
    /**
     * @param string $templatePath
     * @return ClientArea
     */
    protected function prologue(string $templatePath) : \WHMCS\ClientArea
    {
        $clientArea = $this->clientAreaFactory->create();
        $clientArea->initPage();
        $clientArea->requireLogin();
        $clientArea->setPageTitle(\WHMCS\Application\Support\Facades\Lang::trans(self::DEFAULT_PAGE_TITLE));
        $clientArea->addToBreadCrumb('index.php', \WHMCS\Application\Support\Facades\Lang::trans('globalsystemname'));
        $clientArea->addToBreadCrumb('clientarea.php', \WHMCS\Application\Support\Facades\Lang::trans('clientareatitle'));
        $clientArea->addToBreadCrumb(self::TYPE . '.php', \WHMCS\Application\Support\Facades\Lang::trans(self::DEFAULT_PAGE_TITLE));
        $clientArea->setTemplate($templatePath);
        return $clientArea;
    }
    public function confirmPage(\SupportPal\WhmcsIntegration\Request\Request $request) : void
    {
        $clientArea = $this->prologue('../supportpal/supportticketsubmit-confirm');
        $clientArea->setPageTitle($this->locale['tickets_submitted']);
        $clientArea->assign('ticket_number', $request->get('number'));
        $clientArea->assign('ticket_token', $request->get('token'));
        $this->epilogue($clientArea);
    }
    public function selectDepartmentsPage() : void
    {
        $clientArea = $this->prologue('../supportpal/supportticketsubmit-stepone');
        $data = ['parent_id' => 0, 'public' => 1];
        try {
            $departmentsCollection = $this->departmentRepository->findBy($data);
        } catch (\SupportPal\WhmcsIntegration\Exception\InvalidApiResponse $apiResponse) {
            $this->epilogue($clientArea);
            return;
        }
        $clientArea->assign('data', $this->modelToArrayConverter->convertMany($departmentsCollection->getModels()));
        $this->epilogue($clientArea);
    }
    /**
     * @param ClientArea $clientArea
     */
    protected function epilogue(\WHMCS\ClientArea $clientArea) : void
    {
        $baseUrl = $this->modSupportPalRepository->getDefaultBaseUrl();
        $clientArea->assign('supportpalUrl', $baseUrl !== null ? \rtrim($baseUrl, '/') . '/' : '');
        $clientArea->assign('LANG2', $this->locale);
        $clientArea->addOutputHookFunction('ClientAreaPageSubmitTicket');
        $clientArea->output();
    }
    /**
     * @param Request $request
     * @return void
     * @throws InvalidApiResponse
     * @throws InvalidArgumentException
     */
    public function submitTicketPage(\SupportPal\WhmcsIntegration\Request\Request $request) : void
    {
        $clientArea = $this->prologue('../supportpal/supportticketsubmit-steptwo');
        $user = $this->authenticatedUserHelper->getAuthenticatedUser();
        $departmentId = $request->get('deptid', null);
        if ($user === null || $departmentId === null) {
            $this->responseFactory->createRedirectResponse('submitticket.php?step=1')->send();
            return;
        }
        try {
            $department = $this->departmentRepository->find((int) $departmentId);
        } catch (\SupportPal\WhmcsIntegration\Exception\InvalidApiResponse $_) {
            $this->responseFactory->createRedirectResponse('submitticket.php?step=1')->send();
            return;
        }
        $coreSettings = $this->coreSettingsRepository->get();
        $ticketSettings = $this->ticketSettingsRepository->get()->all();
        $ticketSettings['date_format'] = $this->stringHelper->formatDatePHPToJqueryUI((string) $coreSettings->get('date_format', 'd/m/Y'));
        $clientArea->assign('settings', $ticketSettings);
        $channelSettings = $this->channelSettingsRepository->find('web');
        if ($request->get('submit') === '1' && $request->get('message') !== null) {
            try {
                $this->handleCreateTicket($user, $channelSettings, $request, $ticketSettings['default_open_status']);
                return;
            } catch (\SupportPal\WhmcsIntegration\Exception\InvalidCaptchaException $_) {
                $clientArea->assign('submit_status', -1);
            } catch (\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\InvalidArgumentException|\SupportPal\WhmcsIntegration\Exception\InvalidApiResponse $_) {
                $clientArea->assign('submit_status', 0);
            }
            $this->resetTicketFieldValues($clientArea, $request);
        }
        if ($department->getPublic() && $department->getParentId() === null) {
            $clientArea->assign('department', $this->modelToArrayConverter->convertOne($department));
        }
        if ($channelSettings->getShowCaptcha() === '2') {
            $clientArea->assign('recaptcha', $this->captchaHelper->buildHtml());
        }
        $prioritiesFilter = ['department_id' => $department->getId()];
        $prioritiesCollection = $this->priorityRepository->findBy($prioritiesFilter);
        $clientArea->assign('priorities', $this->modelToArrayConverter->convertMany($prioritiesCollection->getModels()));
        $customFields = [];
        $customFieldsFilters = ['department_id' => $departmentId, 'order_column' => 'order'];
        $customFieldsCollection = $this->customFieldRepository->findBy($customFieldsFilters);
        $this->modelToArrayConverter->convertManyAndApply($customFieldsCollection->getModels(), function (array $customField) use(&$customFields) {
            $customFields[$customField['id']] = $customField;
            return $customField;
        });
        $customFields = $this->customFieldsHelper->updateRelatedServiceField($customFields, $clientArea->getUserId());
        foreach ($customFields as $fieldId => $field) {
            // Remove field if not public.
            if (!$field['public']) {
                unset($customFields[$fieldId]);
                continue;
            }
            // Remove field if parent is not public.
            if (isset($field['depends_on_field_id'])) {
                if (!isset($customFields[$field['depends_on_field_id']]) || !$customFields[$field['depends_on_field_id']]['public']) {
                    unset($customFields[$fieldId]);
                    continue;
                }
                $customFields[$fieldId]['parent_selected'] = \false;
            }
            $dependents = $this->customFieldsHelper->getDependentCustomFields($customFields, $field['id']);
            $customFields[$fieldId]['children'] = \json_encode($dependents === \false ? [] : $dependents);
        }
        $clientArea->assign('customfields', \array_values($customFields));
        $clientArea->assign('article_suggestions', $this->modSupportPalRepository->getBool('article_suggestions') ?? 0);
        $this->epilogue($clientArea);
    }
    /**
     * @param User $user
     * @param ChannelSettings $channelSettings
     * @param Request $request
     * @param string $defaultOpenStatus
     * @return void
     * @throws InvalidArgumentException|InvalidApiResponse|InvalidCaptchaException
     */
    private function handleCreateTicket(\WHMCS\User\User $user, \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Ticket\ChannelSettings $channelSettings, \SupportPal\WhmcsIntegration\Request\Request $request, string $defaultOpenStatus) : void
    {
        if ($channelSettings->getShowCaptcha() === '2' && !$this->captchaHelper->verify((string) $request->get('captcha'))) {
            throw new \SupportPal\WhmcsIntegration\Exception\InvalidCaptchaException();
        }
        $ticket = $this->postTicket($user, $request, $defaultOpenStatus);
        $redirectTarget = \sprintf('submitticket.php?step=confirm&number=%s&token=%s', $ticket->getNumber(), \sha1($ticket->getId() . '-' . $ticket->getNumber()));
        $this->responseFactory->createRedirectResponse($redirectTarget)->send();
    }
    /**
     * @param User $user
     * @param Request $request
     * @param string $defaultOpenStatus
     * @return Ticket
     * @throws InvalidArgumentException|InvalidApiResponse
     */
    private function postTicket(\WHMCS\User\User $user, \SupportPal\WhmcsIntegration\Request\Request $request, string $defaultOpenStatus) : \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Ticket\Ticket
    {
        $helpdeskUser = $this->userRepository->getHelpdeskAccount(null, $user->email);
        // Make sure in same organisation as client, otherwise use client account to open (whmcs-integration#51).
        if ($helpdeskUser !== null) {
            $client = $this->authenticatedUserHelper->getAuthenticatedClient();
            // If not same email, continue check.
            if ($client !== null && \strtolower($client->email) !== \strtolower($helpdeskUser['email'])) {
                $helpdeskClient = $this->userRepository->getHelpdeskAccount(null, $client->email);
                // Ensure organisation ID same, else switch to using client account.
                if ($helpdeskClient !== null && (int) $helpdeskUser['organisation_id'] !== (int) $helpdeskClient['organisation_id']) {
                    $helpdeskUser = $helpdeskClient;
                }
            }
        }
        $brandId = $this->modSupportPalRepository->getDefaultBrand();
        $brandId = $brandId === null ? 1 : $brandId;
        $cc = \array_filter((array) $request->get('cc', []), function (string $email) {
            return \filter_var($email, \FILTER_VALIDATE_EMAIL) !== \false;
        });
        $data = ['cc' => $cc, 'department' => (int) $request->get('deptid'), 'brand' => (int) $brandId, 'status' => (int) $defaultOpenStatus, 'priority' => (int) $request->get('urgency'), 'subject' => \WHMCS\Input\Sanitize::decode($request->get('subject')), 'text' => \nl2br($request->get('message')), 'user_ip_address' => $request->getClientIp()];
        if ($helpdeskUser !== null) {
            $data['user'] = $helpdeskUser['id'];
        } else {
            $data['user_firstname'] = $user->first_name;
            $data['user_lastname'] = $user->last_name;
            $data['user_email'] = $user->email;
            if (!empty($user->companyname)) {
                $data['user_organisation'] = $user->companyname;
            }
        }
        $data['customfield'] = [];
        foreach ($request->getBodyArray() as $key => $value) {
            $exploded = \explode('_', $key);
            if (!isset($exploded[0]) || $exploded[0] !== 'cf' || !isset($exploded[1]) || !\is_numeric($exploded[1])) {
                continue;
            }
            $data['customfield'][$exploded[1]] = $value;
        }
        $data = $this->parseAttachments($request, $data);
        return $this->ticketManager->create($data);
    }
    /**
     * @param ClientArea $clientArea
     * @param Request $request
     */
    private function resetTicketFieldValues(\WHMCS\ClientArea $clientArea, \SupportPal\WhmcsIntegration\Request\Request $request) : void
    {
        $cc = $request->get('cc', null);
        if (!empty($cc)) {
            $clientArea->assign('cc', $cc);
        }
        $urgency = $request->get('urgency', null);
        if ($urgency) {
            $clientArea->assign('priority', $urgency);
        }
        $subject = $request->get('subject', null);
        if ($subject) {
            $clientArea->assign('subject', \WHMCS\Input\Sanitize::decode($subject));
        }
        $message = $request->get('message', null);
        if (!$message) {
            return;
        }
        $clientArea->assign('message', \WHMCS\Input\Sanitize::decode($message));
    }
}
