<?php

declare (strict_types=1);
namespace SupportPal\WhmcsIntegration\Controller\Ticket;

use Closure;
use InvalidArgumentException as BaseInvalidArgumentException;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\InvalidArgumentException;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\MissingIdentifierException;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Collection\Collection;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Shared\CustomField;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Shared\Settings;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Ticket\Ticket;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Ticket\TicketCustomField;
use SupportPal\WhmcsIntegration\Exception\InvalidApiResponse;
use SupportPal\WhmcsIntegration\Exception\InvalidTicketException;
use SupportPal\WhmcsIntegration\Factory\ClientAreaFactory;
use SupportPal\WhmcsIntegration\Factory\ResponseFactory;
use SupportPal\WhmcsIntegration\Helper\AuthenticatedUserHelper;
use SupportPal\WhmcsIntegration\Helper\CustomFieldsHelper;
use SupportPal\WhmcsIntegration\Helper\DateTimeFormatter;
use SupportPal\WhmcsIntegration\Helper\StringHelper;
use SupportPal\WhmcsIntegration\Repository\Core\CoreSettingsRepository;
use SupportPal\WhmcsIntegration\Repository\Ticket\CustomFieldRepository;
use SupportPal\WhmcsIntegration\Repository\Ticket\TicketAttachmentRepository;
use SupportPal\WhmcsIntegration\Repository\Ticket\TicketMessageRepository;
use SupportPal\WhmcsIntegration\Repository\Ticket\TicketRepository;
use SupportPal\WhmcsIntegration\Repository\Ticket\TicketSettingsRepository;
use SupportPal\WhmcsIntegration\Repository\UserRepository;
use SupportPal\WhmcsIntegration\Repository\Whmcs\ModSupportPalRepository;
use SupportPal\WhmcsIntegration\Request\Request;
use SupportPal\WhmcsIntegration\Service\Converter\ModelToArrayConverter;
use SupportPal\WhmcsIntegration\Service\Manager\Locale\LocaleManager;
use SupportPal\WhmcsIntegration\Service\Manager\Ticket\MessageManager;
use SupportPal\WhmcsIntegration\Service\Manager\Ticket\TicketManager;
use SupportPal\WhmcsIntegration\Service\Manager\UserManager;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Response;
use WHMCS\Application\Support\Facades\Lang;
use WHMCS\ClientArea;
use function array_filter;
use function array_values;
use function date;
use function explode;
use function filter_var;
use function htmlentities;
use function is_numeric;
use function json_decode;
use function json_encode;
use function md5;
use function nl2br;
use function rtrim;
use function sha1;
use function sprintf;
use const FILTER_VALIDATE_EMAIL;
class ViewTicketController extends \SupportPal\WhmcsIntegration\Controller\Ticket\BaseTicketController
{
    private const DEFAULT_PAGE_TITLE = 'supportticketsviewticket';
    private const TYPE = 'supporttickets';
    /** @var ModSupportPalRepository */
    private $modSupportPalRepository;
    /** @var string[] */
    private $locale;
    /** @var AuthenticatedUserHelper */
    private $authenticatedUserHelper;
    /** @var UserRepository */
    private $userRepository;
    /** @var CoreSettingsRepository */
    private $coreSettingsRepository;
    /** @var StringHelper */
    private $stringHelper;
    /** @var CustomFieldsHelper */
    private $customFieldsHelper;
    /** @var TicketRepository */
    private $ticketRepository;
    /** @var TicketSettingsRepository */
    private $ticketSettingsRepository;
    /** @var ModelToArrayConverter */
    private $modelToArrayConverter;
    /** @var CustomFieldRepository */
    private $customFieldsRepository;
    /** @var TicketMessageRepository */
    private $ticketMessageRepository;
    /** @var MessageManager */
    private $messageManager;
    /** @var TicketManager */
    private $ticketManager;
    /** @var ResponseFactory */
    private $responseFactory;
    /** @var ClientAreaFactory */
    private $clientAreaFactory;
    /** @var TicketAttachmentRepository */
    private $ticketAttachmentRepository;
    /** @var UserManager */
    private $userManager;
    public function __construct(\SupportPal\WhmcsIntegration\Repository\Whmcs\ModSupportPalRepository $modSupportPalRepository, \SupportPal\WhmcsIntegration\Service\Manager\Locale\LocaleManager $localeManager, \SupportPal\WhmcsIntegration\Helper\AuthenticatedUserHelper $authenticatedUserHelper, \SupportPal\WhmcsIntegration\Repository\UserRepository $userRepository, \SupportPal\WhmcsIntegration\Repository\Core\CoreSettingsRepository $coreSettingsRepository, \SupportPal\WhmcsIntegration\Helper\StringHelper $stringHelper, \SupportPal\WhmcsIntegration\Helper\CustomFieldsHelper $customFieldsHelper, \SupportPal\WhmcsIntegration\Repository\Ticket\TicketRepository $ticketRepository, \SupportPal\WhmcsIntegration\Repository\Ticket\TicketSettingsRepository $ticketSettingsRepository, \SupportPal\WhmcsIntegration\Service\Converter\ModelToArrayConverter $modelToArrayConverter, \SupportPal\WhmcsIntegration\Helper\DateTimeFormatter $dateTimeFormatter, \SupportPal\WhmcsIntegration\Repository\Ticket\CustomFieldRepository $customFieldsRepository, \SupportPal\WhmcsIntegration\Repository\Ticket\TicketMessageRepository $ticketMessageRepository, \SupportPal\WhmcsIntegration\Service\Manager\Ticket\MessageManager $messageManager, \SupportPal\WhmcsIntegration\Service\Manager\Ticket\TicketManager $ticketManager, \SupportPal\WhmcsIntegration\Factory\ResponseFactory $responseFactory, \SupportPal\WhmcsIntegration\Factory\ClientAreaFactory $clientAreaFactory, \SupportPal\WhmcsIntegration\Repository\Ticket\TicketAttachmentRepository $ticketAttachmentRepository, \SupportPal\WhmcsIntegration\Service\Manager\UserManager $userManager)
    {
        parent::__construct($dateTimeFormatter);
        $this->modSupportPalRepository = $modSupportPalRepository;
        $this->locale = $localeManager->getLocale(\WHMCS\Application\Support\Facades\Lang::trans('locale'));
        $this->userRepository = $userRepository;
        $this->authenticatedUserHelper = $authenticatedUserHelper;
        $this->coreSettingsRepository = $coreSettingsRepository;
        $this->stringHelper = $stringHelper;
        $this->customFieldsHelper = $customFieldsHelper;
        $this->ticketRepository = $ticketRepository;
        $this->ticketSettingsRepository = $ticketSettingsRepository;
        $this->modelToArrayConverter = $modelToArrayConverter;
        $this->customFieldsRepository = $customFieldsRepository;
        $this->ticketMessageRepository = $ticketMessageRepository;
        $this->messageManager = $messageManager;
        $this->ticketManager = $ticketManager;
        $this->responseFactory = $responseFactory;
        $this->clientAreaFactory = $clientAreaFactory;
        $this->ticketAttachmentRepository = $ticketAttachmentRepository;
        $this->userManager = $userManager;
    }
    public function index(\SupportPal\WhmcsIntegration\Request\Request $request) : void
    {
        $client = $this->authenticatedUserHelper->getAuthenticatedClient();
        $ticketNumber = $request->get('number', null);
        $ticketToken = $request->get('token', null);
        $clientArea = $this->prologue('../supportpal/viewticket');
        if ($client === null || $ticketNumber === null || $ticketToken === null) {
            $this->epilogue($clientArea, \true);
            return;
        }
        $clientArea->addToBreadCrumb('viewticket.php?number=' . \htmlentities($ticketNumber) . '&token=' . \md5($ticketNumber), \WHMCS\Application\Support\Facades\Lang::trans('supportticketsviewticket'));
        try {
            $helpDeskUser = $this->userRepository->getHelpdeskAccount(null, $client->email);
        } catch (\SupportPal\WhmcsIntegration\Exception\InvalidApiResponse $apiResponse) {
            $this->epilogue($clientArea, \true);
            return;
        }
        if ($helpDeskUser === null || empty($helpDeskUser['id'])) {
            $this->epilogue($clientArea, \true);
            return;
        }
        try {
            $response = $this->handle($ticketNumber, $helpDeskUser, $ticketToken, $clientArea, $request);
            if ($response instanceof \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Response) {
                $response->send();
                return;
            }
            $this->epilogue($clientArea, \false);
            return;
        } catch (\SupportPal\WhmcsIntegration\Exception\InvalidApiResponse|\SupportPal\WhmcsIntegration\Exception\InvalidTicketException $_) {
            $this->epilogue($clientArea, \true);
            return;
        }
    }
    protected function prologue(string $templatePath) : \WHMCS\ClientArea
    {
        $clientArea = $this->clientAreaFactory->create();
        $clientArea->initPage();
        $clientArea->requireLogin();
        $clientArea->setTemplate($templatePath);
        $clientArea->setPageTitle(\WHMCS\Application\Support\Facades\Lang::trans(self::DEFAULT_PAGE_TITLE));
        $clientArea->addToBreadCrumb('index.php', \WHMCS\Application\Support\Facades\Lang::trans('globalsystemname'));
        $clientArea->addToBreadCrumb('clientarea.php', \WHMCS\Application\Support\Facades\Lang::trans('clientareatitle'));
        $clientArea->addToBreadCrumb(self::TYPE . '.php', \WHMCS\Application\Support\Facades\Lang::trans('supportticketspagetitle'));
        return $clientArea;
    }
    protected function epilogue(\WHMCS\ClientArea $clientArea, bool $ticketError) : void
    {
        $clientArea->assign('ticketError', $ticketError);
        $baseUrl = $this->modSupportPalRepository->getDefaultBaseUrl();
        $clientArea->assign('supportpalUrl', $baseUrl !== null ? \rtrim($baseUrl, '/') . '/' : '');
        $clientArea->assign('LANG2', $this->locale);
        $clientArea->addOutputHookFunction('ClientAreaPageViewTicket');
        $clientArea->output();
    }
    /**
     * @param string $ticketNumber
     * @param mixed[] $helpdeskUser
     * @param string $ticketToken
     * @param ClientArea $ca
     * @param Request $request
     * @return Response|null
     * @throws InvalidApiResponse
     * @throws InvalidArgumentException
     * @throws InvalidTicketException
     */
    private function handle(string $ticketNumber, array $helpdeskUser, string $ticketToken, \WHMCS\ClientArea $ca, \SupportPal\WhmcsIntegration\Request\Request $request) : ?\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Response
    {
        $data = ['number' => $ticketNumber];
        // Handle if the user is part of an organisation
        if (!empty($helpdeskUser['organisation_id']) && (int) $helpdeskUser['organisation_access_level'] === 0) {
            $data['organisation'] = $helpdeskUser['organisation_id'];
        } else {
            $data['user'] = $helpdeskUser['id'];
        }
        $ticketCollection = $this->ticketRepository->findBy($data);
        if ($ticketCollection->isEmpty()) {
            throw new \SupportPal\WhmcsIntegration\Exception\InvalidTicketException();
        }
        /** @var Ticket|null $ticket */
        $ticket = $ticketCollection->filter(function (\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Ticket\Ticket $ticket) use($ticketNumber) {
            return $ticket->getNumber() === $ticketNumber;
        })->first();
        if ($ticket === null || $ticketToken !== \sha1($ticket->getId() . '-' . $ticket->getNumber())) {
            throw new \SupportPal\WhmcsIntegration\Exception\InvalidTicketException();
        }
        $coreSettings = $this->coreSettingsRepository->get();
        $ticketSettings = $this->ticketSettingsRepository->get();
        $settings = $ticketSettings->all();
        $settings['date_format'] = $this->stringHelper->formatDatePHPToJqueryUI((string) $coreSettings->get('date_format', 'd/m/Y'));
        $ca->assign('settings', $settings);
        $downloadFile = $request->get('download_file', null);
        if ($downloadFile !== null) {
            return $this->downloadAttachment((int) $downloadFile, (int) $ticket->getId());
        }
        if ($request->get('reply', null) === '1') {
            return $this->postMessage($ticket, $ticketNumber, $ticketToken, $request, $helpdeskUser['organisation_id']);
        }
        if ($request->get('close', null) === '1') {
            return $this->closeTicket((string) $ticketSettings->get('default_resolved_status'), $ticket, $ticketNumber, $ticketToken);
        }
        if ($request->get('update_fields', null) === '1') {
            return $this->updateCustomFields($ticket, (int) $ticketSettings->get('default_resolved_status'), $ticketNumber, $ticketToken, $request);
        }
        $ticketData = $this->modelToArrayConverter->convertOneAndApply($ticket, $this->setTicketToken($ticketToken), $this->formatDateTime($coreSettings));
        $data = ['department_id' => $ticket->getDepartmentId(), 'order_column' => 'order'];
        $customFieldsCollection = $this->customFieldsRepository->findBy($data);
        $customFields = $this->getCustomFields($customFieldsCollection, $ticket, $coreSettings);
        // Update custom fields if related service field is set
        $customFields = $this->customFieldsHelper->updateRelatedServiceField($customFields, $ca->getUserID());
        foreach ($customFields as $fieldId => $field) {
            // Remove field if not public.
            if (!$field['public']) {
                unset($customFields[$fieldId]);
                continue;
            }
            // Remove field if parent is not public.
            if (isset($field['depends_on_field_id'])) {
                if (!isset($customFields[$field['depends_on_field_id']]) || !$customFields[$field['depends_on_field_id']]['public']) {
                    unset($customFields[$fieldId]);
                    continue;
                }
                // Determine if the parent has a value.
                $customFields[$fieldId]['parent_selected'] = $this->customFieldsHelper->isCustomFieldParentSelected($customFields, $field);
            }
            // Determine children of each field.
            $dependents = $this->customFieldsHelper->getDependentCustomFields($customFields, $field['id']);
            $customFields[$fieldId]['children'] = \json_encode($dependents === \false ? [] : $dependents);
        }
        // Send custom fields to view with keys reset
        $ca->assign('customfields', \array_values($customFields));
        // Get messages
        $ticketMessagesCollection = $this->getAllTicketMessages($ticket, $ticketSettings);
        if (!$ticketMessagesCollection->isEmpty()) {
            $messages = $this->modelToArrayConverter->convertManyAndApply($ticketMessagesCollection->getModels(), $this->formatCreatedAt($coreSettings));
            $ca->assign('messages', $messages);
        }
        // Send all data to view
        $ca->assign('data', $ticketData);
        return null;
    }
    /**
     * @param Ticket $ticket
     * @param Settings $settings
     * @return Collection
     * @throws InvalidArgumentException
     * @throws InvalidApiResponse
     */
    private function getAllTicketMessages(\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Ticket\Ticket $ticket, \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Shared\Settings $settings) : \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Collection\Collection
    {
        $data = ['order_column' => 'created_at', 'order_direction' => (int) $settings->get('ticket_reply_order') === 1 ? 'desc' : 'asc', 'start' => 1];
        $collection = new \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Collection\Collection();
        while (\true) {
            $ticketMessagesCollection = $this->ticketMessageRepository->findBy($ticket->getId(), $data);
            if ($ticketMessagesCollection->isEmpty()) {
                break;
            }
            $data['start'] += $ticketMessagesCollection->getModelsCount();
            $collection = $collection->merge($ticketMessagesCollection);
            /**
             * no need for an extra api call that would return an empty response
             */
            if ($collection->getModelsCount() === $ticketMessagesCollection->getCount()) {
                break;
            }
        }
        return $collection;
    }
    /**
     * @param Ticket $ticket
     * @param string $ticketNumber
     * @param string $ticketToken
     * @param Request $request
     * @param int|null $organisationId
     * @return Response
     * @throws InvalidArgumentException
     */
    private function postMessage(\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Ticket\Ticket $ticket, string $ticketNumber, string $ticketToken, \SupportPal\WhmcsIntegration\Request\Request $request, ?int $organisationId = null) : \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Response
    {
        $redirectionTarget = \sprintf('viewticket.php?number=%s&token=%s', $ticketNumber, $ticketToken);
        $cc = \array_filter($request->get('cc', []), function ($email) {
            return \filter_var($email, \FILTER_VALIDATE_EMAIL) !== \false;
        });
        try {
            // We need to do this in case the user is different to the client account (multi-account support).
            $user = $this->authenticatedUserHelper->getAuthenticatedUser();
            if ($user === null) {
                throw new \InvalidArgumentException();
            }
            $helpDeskUser = $this->userRepository->getHelpdeskAccount(null, $user->email);
            // Does user exist?
            if ($helpDeskUser === null || empty($helpDeskUser['id'])) {
                // Create a user
                $user = $this->userManager->create(['firstname' => $user->first_name, 'lastname' => $user->last_name, 'email' => $user->email, 'organisation_id' => $organisationId]);
                $helpDeskUser = ['id' => $user->getId()];
            }
            $data = ['ticket_id' => $ticket->getId(), 'user_id' => $helpDeskUser['id'], 'user_ip_address' => $request->getClientIp(), 'message_type' => 0, 'text' => \nl2br($request->get('replymessage')), 'cc' => $cc];
            $data = $this->parseAttachments($request, $data);
            $this->messageManager->create($data);
            return $this->responseFactory->createRedirectResponse($redirectionTarget . '&reply_posted=1');
        } catch (\SupportPal\WhmcsIntegration\Exception\InvalidApiResponse|\InvalidArgumentException $_) {
            return $this->responseFactory->createRedirectResponse($redirectionTarget . '&reply_posted=0');
        }
    }
    /**
     * @param string $defaultResolvedStatus
     * @param Ticket $ticket
     * @param string $ticketNumber
     * @param string $ticketToken
     * @return Response
     * @throws InvalidArgumentException
     */
    private function closeTicket(string $defaultResolvedStatus, \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Ticket\Ticket $ticket, string $ticketNumber, string $ticketToken) : \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Response
    {
        $data = ['status' => $defaultResolvedStatus];
        $redirectionTarget = \sprintf('viewticket.php?number=%s&token=%s', $ticketNumber, $ticketToken);
        try {
            $this->ticketManager->update($ticket, $data);
            return $this->responseFactory->createRedirectResponse($redirectionTarget . '&user_closed=1');
        } catch (\SupportPal\WhmcsIntegration\Exception\InvalidApiResponse $_) {
            return $this->responseFactory->createRedirectResponse($redirectionTarget . '&user_closed=0');
        }
    }
    /**
     * @param Ticket $ticket
     * @param int $defaultResolvedStatus
     * @param string $ticketNumber
     * @param string $ticketToken
     * @param Request $request
     * @return Response
     * @throws InvalidArgumentException
     */
    private function updateCustomFields(\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Ticket\Ticket $ticket, int $defaultResolvedStatus, string $ticketNumber, string $ticketToken, \SupportPal\WhmcsIntegration\Request\Request $request) : ?\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Response
    {
        if ($ticket->getStatusId() === $defaultResolvedStatus || $ticket->getLocked()) {
            return null;
        }
        $data = ['customfield' => []];
        foreach ($request->getBodyArray() as $key => $value) {
            $exploded = \explode('_', $key);
            if (!isset($exploded[0]) || $exploded[0] !== 'cf' || !isset($exploded[1]) || !\is_numeric($exploded[1]) || $value === '** ENCRYPTED **') {
                continue;
            }
            $data['customfield'][$exploded[1]] = $value;
        }
        $redirectionTarget = \sprintf('viewticket.php?number=%s&token=%s', $ticketNumber, $ticketToken);
        try {
            $this->ticketManager->update($ticket, $data);
            return $this->responseFactory->createRedirectResponse($redirectionTarget . '&saved_fields=1');
        } catch (\SupportPal\WhmcsIntegration\Exception\InvalidApiResponse $_) {
            return $this->responseFactory->createRedirectResponse($redirectionTarget . '&saved_fields=0');
        }
    }
    /**
     * @param string $ticketToken
     * @return Closure
     */
    private function setTicketToken(string $ticketToken) : \Closure
    {
        return function (array $ticket) use($ticketToken) {
            $ticket['token'] = $ticketToken;
            return $ticket;
        };
    }
    /**
     * @param Settings $coreSettings
     * @return Closure
     */
    private function formatCreatedAt(\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Shared\Settings $coreSettings) : \Closure
    {
        return function (array $message) use($coreSettings) {
            $message['created_at'] = $this->dateTimeFormatter->format((string) $coreSettings->get('date_format', 'd/m/Y'), (string) $coreSettings->get('time_format', 'g:i A'), $message['created_at']);
            return $message;
        };
    }
    /**
     * @param Collection $customFieldsCollection
     * @param Ticket $ticket
     * @param Settings $coreSettings
     * @return mixed[]
     */
    private function getCustomFields(\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Collection\Collection $customFieldsCollection, \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Ticket\Ticket $ticket, \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Shared\Settings $coreSettings) : array
    {
        $ticketCustomFieldsCluster = $this->clusterByFieldId($ticket);
        $customFields = [];
        /** @var CustomField $customField */
        foreach ($customFieldsCollection->getModels() as $customField) {
            $ticketCustomField = $ticketCustomFieldsCluster[$customField->getId()];
            $value = $customField->getValue();
            /**
             * value is set for this custom field
             */
            if ($ticketCustomField !== null) {
                $value = $this->getDisplayValue($ticketCustomField, $customField, $coreSettings);
            }
            $modelArray = $this->modelToArrayConverter->convertOne($customField);
            $modelArray['value'] = $value;
            $customFields[$customField->getId()] = $modelArray;
        }
        return $customFields;
    }
    /**
     * Download a file from a give URL in chunks
     *
     * @param int $id
     * @param int $ticketId
     * @return Response
     * @throws InvalidApiResponse
     * @throws MissingIdentifierException
     */
    private function downloadAttachment(int $id, int $ticketId) : ?\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Response
    {
        $attachment = $this->ticketAttachmentRepository->find($id);
        if ($attachment->getTicketId() !== $ticketId) {
            return null;
        }
        $attachmentStream = $this->ticketAttachmentRepository->download($attachment);
        return $this->responseFactory->createDownloadResponse($attachmentStream, $attachment->getOriginalName());
    }
    /**
     * @param Ticket $ticket
     * @return mixed[]
     */
    private function clusterByFieldId(\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Ticket\Ticket $ticket) : array
    {
        $cluster = [];
        foreach ((array) $ticket->getCustomfields() as $ticketCustomField) {
            $cluster[$ticketCustomField->getFieldId()] = $ticketCustomField;
        }
        return $cluster;
    }
    /**
     * @param TicketCustomField $ticketCustomField
     * @param CustomField $customField
     * @param Settings $coreSettings
     * @return string|mixed[]|null
     */
    private function getDisplayValue(\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Ticket\TicketCustomField $ticketCustomField, \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Shared\CustomField $customField, \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Shared\Settings $coreSettings)
    {
        if ($customField->getEncrypted()) {
            return '** ENCRYPTED **';
        }
        $value = $ticketCustomField->getValue();
        switch ($customField->getType()) {
            case \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Shared\CustomField::TYPE_CHECKLIST:
            case \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Shared\CustomField::TYPE_MULTIPLE:
                return \json_decode((string) $value);
            case \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Shared\CustomField::TYPE_DATE:
                return \date((string) $coreSettings->get('date_format', 'd/m/Y'), (int) $value);
            default:
                return $value;
        }
    }
}
