<?php

declare (strict_types=1);
namespace SupportPal\WhmcsIntegration\Helper;

use WHMCS\Database\Capsule;
use function array_merge;
class CustomFieldsHelper
{
    /**
     * Get the children of a given custom field (dependent fields).
     * @TODO proper type hinting should be added, alongside throwing an exception if recursion depth is exceeded
     * @param  mixed[] $collection
     * @param  int     $parentId
     * @param  int     $recursion [Internal] Do not modify.
     * @return mixed[]|false FALSE if 100 nesting levels is reached.
     */
    public function getDependentCustomFields(array $collection, int $parentId, int $recursion = 0)
    {
        $children = [];
        if ($recursion >= 100) {
            return \false;
        }
        foreach ($collection as $key => $element) {
            if (!isset($element['depends_on_field_id']) || $element['depends_on_field_id'] !== $parentId) {
                continue;
            }
            $children[] = $element['id'];
            $ids = $this->getDependentCustomFields($collection, $element['id'], ++$recursion);
            if ($ids === \false) {
                return \false;
            }
            $children = \array_merge($children, $ids);
        }
        return $children;
    }
    /**
     * Determine whether the given custom field's parent is selected.
     *
     * @param  mixed[] $customFields  Array of custom fields indexed by field ID, also includes value.
     * @param  mixed[] $customField   Current custom field we're considering
     * @return bool
     */
    public function isCustomFieldParentSelected(array $customFields, array $customField) : bool
    {
        while (isset($customField['depends_on_option_id'])) {
            $result = isset($customFields[$customField['depends_on_field_id']]) && $customFields[$customField['depends_on_field_id']]['value'] === $customField['depends_on_option_id'];
            // The field's parent is not selected, so none of it's children should be.
            if (!$result) {
                return \false;
            }
            // Work back up the tree until we find the root.
            $customField = $customFields[$customField['depends_on_field_id']];
        }
        return \true;
    }
    /**
     * @TODO this function should not depend on the database, values should be passed as arguments
     *
     * Updates the custom field options if the related service field is set and is present
     * in the custom fields array.
     *
     * @param   mixed[]  $customFields
     * @param   int    $userId
     * @return  mixed[]
     */
    public function updateRelatedServiceField(array $customFields, int $userId = 0) : array
    {
        // Check if we have the related product/service field
        $relatedField = \WHMCS\Database\Capsule::table('mod_supportpal')->where('name', 'related_service_field')->first(array('value'));
        // If we do, and the custom field is visible in our array, update the options
        if (!empty($relatedField) && !empty($relatedField->value) && isset($customFields[$relatedField->value])) {
            // Fetch client's products, ordered by product name and domain
            $products = \WHMCS\Database\Capsule::table('tblhosting')->join('tblproducts', 'tblhosting.packageid', '=', 'tblproducts.id')->where('userid', $userId)->orderBy('tblproducts.name')->orderBy('domain')->get(array('tblhosting.id', 'tblproducts.name', 'domain', 'domainstatus'));
            // Do we have any products?
            if ($products->isNotEmpty()) {
                // Convert to formatted array for UI
                $productOptions = array();
                foreach ($products as $product) {
                    $productOptions[] = ['id' => $product->id, 'value' => $product->name . (!empty($product->domain) ? ' - ' . $product->domain : '') . ' (' . $product->domainstatus . ')'];
                }
                // Update options to our product list
                $customFields[$relatedField->value]['options'] = $productOptions;
            } else {
                // No products, so don't show custom field
                unset($customFields[$relatedField->value]);
            }
        }
        return $customFields;
    }
}
