<?php

declare (strict_types=1);
namespace SupportPal\WhmcsIntegration;

use Closure;
use Smarty;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Ticket\Status;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Ticket\Ticket;
use SupportPal\WhmcsIntegration\Exception\DefaultResolvedStatusNotFound;
use SupportPal\WhmcsIntegration\Exception\InvalidApiResponse;
use SupportPal\WhmcsIntegration\Factory\RequestFactory;
use SupportPal\WhmcsIntegration\Helper\AuthenticatedUserHelper;
use SupportPal\WhmcsIntegration\Helper\SessionHelper;
use SupportPal\WhmcsIntegration\Repository\Ticket\StatusRepository;
use SupportPal\WhmcsIntegration\Repository\Ticket\TicketRepository;
use SupportPal\WhmcsIntegration\Repository\Ticket\TicketSettingsRepository;
use SupportPal\WhmcsIntegration\Repository\UserRepository;
use SupportPal\WhmcsIntegration\Repository\Whmcs\ClientRepository;
use SupportPal\WhmcsIntegration\Repository\Whmcs\ModSupportPalRepository;
use SupportPal\WhmcsIntegration\Repository\Whmcs\WhmcsUserRepository;
use SupportPal\WhmcsIntegration\Service\Manager\WhmcsSupportPalUserClientOrchestrator;
use WHMCS\Application;
use WHMCS\Application\Support\Facades\Lang;
use WHMCS\User\Relations\UserClient;
use WHMCS\User\User;
use WHMCS\View\Menu\Item;
use function explode;
use function htmlentities;
use function implode;
use function in_array;
use function rtrim;
use function sha1;
use function sprintf;
class HooksEncapsulator
{
    /** @var WhmcsSupportPalUserClientOrchestrator */
    private $whmcsSupportPalUserClientOrchestrator;
    /** @var AuthenticatedUserHelper */
    private $authenticatedUserHelper;
    /** @var SessionHelper */
    private $sessionHelper;
    /** @var UserRepository */
    private $userRepository;
    /** @var StatusRepository */
    private $statusRepository;
    /** @var TicketSettingsRepository */
    private $ticketSettingsRepository;
    /** @var TicketRepository */
    private $ticketRepository;
    /** @var ModSupportPalRepository */
    private $modSupportPalRepository;
    /** @var ClientRepository */
    private $clientRepository;
    /** @var WhmcsUserRepository */
    private $whmcsUserRepository;
    /** @var RequestFactory */
    private $requestFactory;
    /**
     * HooksEncapsulator constructor.
     * @param WhmcsSupportPalUserClientOrchestrator $whmcsSupportPalUserClientOrchestrator
     * @param SessionHelper $sessionHelper
     * @param AuthenticatedUserHelper $authenticatedUserHelper
     * @param UserRepository $userRepository
     * @param StatusRepository $statusRepository
     * @param TicketSettingsRepository $ticketSettingsRepository
     * @param TicketRepository $ticketRepository
     * @param ModSupportPalRepository $modSupportPalRepository
     * @param ClientRepository $clientRepository
     * @param WhmcsUserRepository $whmcsUserRepository
     * @param RequestFactory $requestFactory
     */
    public function __construct(\SupportPal\WhmcsIntegration\Service\Manager\WhmcsSupportPalUserClientOrchestrator $whmcsSupportPalUserClientOrchestrator, \SupportPal\WhmcsIntegration\Helper\SessionHelper $sessionHelper, \SupportPal\WhmcsIntegration\Helper\AuthenticatedUserHelper $authenticatedUserHelper, \SupportPal\WhmcsIntegration\Repository\UserRepository $userRepository, \SupportPal\WhmcsIntegration\Repository\Ticket\StatusRepository $statusRepository, \SupportPal\WhmcsIntegration\Repository\Ticket\TicketSettingsRepository $ticketSettingsRepository, \SupportPal\WhmcsIntegration\Repository\Ticket\TicketRepository $ticketRepository, \SupportPal\WhmcsIntegration\Repository\Whmcs\ModSupportPalRepository $modSupportPalRepository, \SupportPal\WhmcsIntegration\Repository\Whmcs\ClientRepository $clientRepository, \SupportPal\WhmcsIntegration\Repository\Whmcs\WhmcsUserRepository $whmcsUserRepository, \SupportPal\WhmcsIntegration\Factory\RequestFactory $requestFactory)
    {
        $this->whmcsSupportPalUserClientOrchestrator = $whmcsSupportPalUserClientOrchestrator;
        $this->sessionHelper = $sessionHelper;
        $this->authenticatedUserHelper = $authenticatedUserHelper;
        $this->userRepository = $userRepository;
        $this->statusRepository = $statusRepository;
        $this->ticketSettingsRepository = $ticketSettingsRepository;
        $this->ticketRepository = $ticketRepository;
        $this->modSupportPalRepository = $modSupportPalRepository;
        $this->clientRepository = $clientRepository;
        $this->whmcsUserRepository = $whmcsUserRepository;
        $this->requestFactory = $requestFactory;
    }
    /**
     * @return array<string, string[]>
     */
    public function getHooks() : array
    {
        // Order of the array dictates priority.
        return ['ClientAdd' => ['clientAdd'], 'ClientEdit' => ['clientEdit'], 'ClientChangePassword' => ['clientChangePassword'], 'UserAdd' => ['userAdd'], 'UserChangePassword' => ['userChangePassword'], 'ClientAreaPageHome' => ['clientAreaPageHome'], 'ClientAreaSecondarySidebar' => ['clientAreaSecondarySidebar'], 'ClientAreaPrimaryNavbar' => ['clientAreaPrimaryNavbar'], 'ClientAreaFooterOutput' => ['clientAreaFooterOutput'], 'AdminAreaFooterOutput' => ['adminAreaFooterOutput']];
    }
    /**
     * @return array<string, array<string, string[]>>
     */
    public function getModelEvents() : array
    {
        return [\WHMCS\User\Relations\UserClient::class => ['saved' => ['onUserClientSaved'], 'deleted' => ['onUserClientDeleted']]];
    }
    /**
     * Hook to handle when a new user is added to WHMCS.
     * It will attempt to add or update the user in the help desk.
     *
     * @param array<mixed> $vars
     */
    public function clientAdd(array $vars) : void
    {
        $this->callAndCatchException(function () use($vars) {
            $this->whmcsSupportPalUserClientOrchestrator->addClient($vars);
        });
    }
    /**
     * Hook to handle when a user is updated in WHMCS.
     * It will attempt to update the user in the help desk.
     *
     * @param array<mixed> $vars
     */
    public function clientEdit(array $vars) : void
    {
        $this->callAndCatchException(function () use($vars) {
            $this->whmcsSupportPalUserClientOrchestrator->editClient($vars);
        });
    }
    /**
     * Hook to handle when a user updated their password.
     *
     * @param array<mixed> $vars
     */
    public function clientChangePassword(array $vars) : void
    {
        $client = $this->clientRepository->getOneById((int) $vars['userid']);
        if ($client === null) {
            return;
        }
        $this->changePassword($client->email, $vars['password']);
    }
    /**
     * Hook to handle when a user is added to WHMCS.
     * It will simply register an account or update their profile and password.
     *
     * @param array<mixed> $vars
     */
    public function userAdd(array $vars) : void
    {
        $this->callAndCatchException(function () use($vars) {
            $this->whmcsSupportPalUserClientOrchestrator->userAdd($vars);
        });
    }
    /**
     * Hook to handle when a user updated their password.
     *
     * @param array<mixed> $vars
     */
    public function userChangePassword(array $vars) : void
    {
        /** @var User|null $user */
        $user = $this->whmcsUserRepository->getOneById((int) $vars['userid']);
        if ($user === null) {
            return;
        }
        $this->changePassword($user->email, $vars['password']);
    }
    /**
     * Model event to handle when a user (sub-account) is linked to a client account.
     * It will attempt to add a manager to main client's organisation.
     *
     * @param UserClient $model
     */
    public function onUserClientSaved(\WHMCS\User\Relations\UserClient $model) : void
    {
        // Only run event once.
        if (isset($GLOBALS['supportpal_userclient_event_has_ran'])) {
            return;
        }
        // Permissions are not always set and result in a LogicException if accessed when unavailable.
        $attributes = $model->getAttributes();
        if (!isset($attributes['permissions'])) {
            return;
        }
        // Check if they have the tickets permission
        $permissions = \explode(',', $attributes['permissions']);
        if (!\in_array('tickets', $permissions)) {
            return;
        }
        $this->callAndCatchException(function () use($model) {
            $this->whmcsSupportPalUserClientOrchestrator->onUserClientSaved($model);
        });
        // Don't let it run the handler again needlessly.
        $GLOBALS['supportpal_userclient_event_has_ran'] = \true;
    }
    /**
     * Model event to handle when a user (sub-account) is unlinked from a client account.
     * It will attempt to remove from the user from the main client's organisation.
     *
     * @param UserClient $model
     */
    public function onUserClientDeleted(\WHMCS\User\Relations\UserClient $model) : void
    {
        $this->callAndCatchException(function () use($model) {
            $this->whmcsSupportPalUserClientOrchestrator->onUserClientDeleted($model);
        });
    }
    /**
     * Hook to handle when viewing the client area home page
     * It will attempt to replace the tickets list and count - it's important
     * this function returns the $var data which has been manipulated.
     *
     * @param array<mixed> $vars
     * @return array<mixed>
     */
    public function clientAreaPageHome(array $vars) : array
    {
        /** @var Item<Item>|null $panels */
        $panels = $vars['panels'] ?? null;
        $uid = $this->sessionHelper->get('uid');
        /** Is user (or sub-account) logged in
         * Only carry on if we're on the client area home which has panels
         **/
        if ($uid === null || $panels === null) {
            return $vars;
        }
        $this->callAndCatchException(function () use(&$vars, $panels) {
            // Ensure they have the tickets permission.
            if (!$this->authenticatedUserHelper->hasTicketsPermission()) {
                $vars['clientsstats']['numtickets'] = $vars['clientsstats']['numactivetickets'] = 0;
                return;
            }
            $client = $this->authenticatedUserHelper->getAuthenticatedClient();
            if ($client === null) {
                return;
            }
            $helpdeskUser = $this->userRepository->getHelpdeskAccount(null, $client->email);
            if ($helpdeskUser === null) {
                return;
            }
            $unresolvedStatuses = [];
            $ticketStatuses = $this->statusRepository->findBy([]);
            if (!$ticketStatuses->isEmpty()) {
                $ticketSettings = $this->ticketSettingsRepository->get();
                $defaultResolvedStatus = $ticketSettings->get('default_resolved_status');
                if ($defaultResolvedStatus === null) {
                    throw new \SupportPal\WhmcsIntegration\Exception\DefaultResolvedStatusNotFound();
                }
                /** @var Status $status */
                foreach ($ticketStatuses->getModels() as $status) {
                    $statusId = (string) $status->getId();
                    if ($statusId === $defaultResolvedStatus) {
                        continue;
                    }
                    $unresolvedStatuses[] = $statusId;
                }
            }
            /** Get all tickets from user that aren't resolved **/
            $data = ['internal' => 0, 'status' => !empty($unresolvedStatuses) ? \implode(',', $unresolvedStatuses) : -1, 'limit' => 5, 'order_column' => 'updated_at', 'order_direction' => 'desc'];
            if (!empty($helpdeskUser['organisation_id']) && (int) $helpdeskUser['organisation_access_level'] === 0) {
                $data['organisation'] = $helpdeskUser['organisation_id'];
            } else {
                $data['user'] = $helpdeskUser['id'];
            }
            $ticketsCollection = $this->ticketRepository->findBy($data);
            /** @var Item|null $whmcsTickets */
            $whmcsTickets = $panels->getChild('Recent Support Tickets');
            if (isset($whmcsTickets)) {
                // Hack to set ticket lists on client area home
                // Clear existing list
                $whmcsTickets->setChildren([]);
                if (!$ticketsCollection->isEmpty()) {
                    $whmcsTickets->setBodyHtml('');
                    /** @var Ticket $ticket */
                    foreach ($ticketsCollection->getModels() as $ticket) {
                        $ticketNumber = \htmlentities($ticket->getNumber());
                        $ticketSubject = \htmlentities($ticket->getSubject());
                        $ticketId = $ticket->getId();
                        $ticketToken = \sha1($ticketId . '-' . $ticketNumber);
                        $whmcsTickets->addChild('ticket-' . $ticketNumber, ['uri' => \sprintf('viewticket.php?number=%s&token=%s', $ticketNumber, $ticketToken), 'label' => \sprintf('#%s - %s', $ticketNumber, $ticketSubject)]);
                    }
                } else {
                    $whmcsTickets->setBodyHtml('<p>' . \WHMCS\Application\Support\Facades\Lang::trans('clientHomePanels.recentSupportTicketsNone') . '</p>');
                }
            }
            $vars['clientsstats']['numtickets'] = $vars['clientsstats']['numactivetickets'] = $ticketsCollection->getCount();
        });
        // Remove news panel.
        $panels->removeChild('Recent News');
        return $vars;
    }
    /**
     * Change items in secondary sidebar.
     * @param Item<Item> $secondarySidebar
     */
    public function clientAreaSecondarySidebar(\WHMCS\View\Menu\Item $secondarySidebar) : void
    {
        if ($secondarySidebar->getChild('Support') === null) {
            return;
        }
        $secondarySidebar = $secondarySidebar->getChild('Support');
        // Update links back to old format
        if (!empty($secondarySidebar->getChild('Announcements'))) {
            $secondarySidebar->getChild('Announcements')->setUri('announcements.php');
        }
        if (!empty($secondarySidebar->getChild('Knowledgebase'))) {
            $secondarySidebar->getChild('Knowledgebase')->setUri('knowledgebase.php');
        }
        if (empty($secondarySidebar->getChild('Downloads'))) {
            return;
        }
        $secondarySidebar->getChild('Downloads')->setUri('downloads.php');
    }
    /**
     * Change items in the navigation bar.
     *
     * @param Item<Item> $primaryNavbar
     */
    public function clientAreaPrimaryNavbar(\WHMCS\View\Menu\Item $primaryNavbar) : void
    {
        // Logged in view
        if ($primaryNavbar->getChild('Support') !== null) {
            $primaryNavbar = $primaryNavbar->getChild('Support');
        }
        // Update links back to old format
        if (!empty($primaryNavbar->getChild('Announcements'))) {
            $primaryNavbar->getChild('Announcements')->setUri('announcements.php');
        }
        if (!empty($primaryNavbar->getChild('Knowledgebase'))) {
            $primaryNavbar->getChild('Knowledgebase')->setUri('knowledgebase.php');
        }
        if (empty($primaryNavbar->getChild('Downloads'))) {
            return;
        }
        $primaryNavbar->getChild('Downloads')->setUri('downloads.php');
    }
    /**
     * Load our footers.
     *
     * @param array<mixed> $vars
     * @return string
     */
    public function clientAreaFooterOutput(array $vars) : string
    {
        /** @var Smarty $smarty */
        $smarty = $GLOBALS['smarty'];
        $basePath = __DIR__ . '/../../../templates/supportpal';
        switch ($vars['templatefile']) {
            case 'clientareahome':
                return $smarty->fetch($basePath . '/clientareahome_footer.tpl');
            case '../supportpal/viewticket':
                return $smarty->fetch($basePath . '/viewticket_footer.tpl');
            case '../supportpal/supportticketsubmit-steptwo':
                return $smarty->fetch($basePath . '/supportticketsubmit-steptwo_footer.tpl');
            case '../supportpal/announcement_view':
            case '../supportpal/knowledgebase_view':
            case '../supportpal/download_view':
                return $smarty->fetch($basePath . '/article_view_footer.tpl');
            default:
                return '';
        }
    }
    /**
     * Change support links on client admin pages.
     *
     * @param array<mixed> $vars
     * @return string
     */
    public function adminAreaFooterOutput(array $vars) : string
    {
        // These must be set to ensure we're on a client page.
        if (empty($GLOBALS['clientsdetails']) || !isset($GLOBALS['clientsdetails']['userid'])) {
            return '';
        }
        try {
            $request = $this->requestFactory->create();
            $helpdeskUser = $this->userRepository->getHelpdeskAccount((int) $request->get('userid'));
        } catch (\SupportPal\WhmcsIntegration\Exception\InvalidApiResponse $invalidApiResponse) {
            return '';
        }
        if (empty($helpdeskUser)) {
            return '';
        }
        $adminUrl = $this->modSupportPalRepository->get('admin_url');
        $adminUrl = $adminUrl !== null ? \rtrim($adminUrl, '/') : '';
        // Create the two links needed.
        $ticketsUrl = $adminUrl . '/user/manage/' . $helpdeskUser['id'] . '/ticket';
        $submitUrl = $adminUrl . '/ticket/create/step1?user=' . $helpdeskUser['id'];
        // Update the links.
        return <<<HTML
    <script type="text/javascript">
        \$(document).ready(function () {
            \$('ul.nav.client-tabs li.tab a[href\$=\\"/tickets\\"], .clientssummarybox a[href^=\\"supporttickets.php?view=any\\"]')
                .attr('href', '{$ticketsUrl}')
                .attr('target', '_blank');
            \$('.clientssummarybox a[href^=\\"supporttickets.php?action=open\\"]')
                .attr('href', '{$submitUrl}')
                .attr('target', '_blank');
        });
    </script>
HTML;
    }
    /**
     * @param Application $application
     * @param string $section
     * @return void
     */
    public function redirectToSupportPalPage(\WHMCS\Application $application, string $section) : void
    {
        if (!$this->modSupportPalRepository->selfServiceTypeEnabled($section)) {
            return;
        }
        $application->redirectSystemURL($section . '.php');
    }
    /**
     * @param Closure $workClosure
     * @return mixed
     */
    private function callAndCatchException(\Closure $workClosure)
    {
        try {
            return $workClosure();
        } catch (\SupportPal\WhmcsIntegration\Exception\InvalidApiResponse $invalidApiResponse) {
            return null;
        }
    }
    /**
     * @param string $email
     * @param string $password
     */
    private function changePassword(string $email, string $password) : void
    {
        $this->callAndCatchException(function () use($email, $password) {
            $this->whmcsSupportPalUserClientOrchestrator->updateClientPassword($email, $password);
        });
    }
}
