<?php

declare (strict_types=1);
namespace SupportPal\WhmcsIntegration\Request;

use Illuminate\Support\Str;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\FileBag;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Request as SymfonyRequest;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\ServerBag;
use WHMCS\Input\Sanitize;
use function is_array;
use function is_string;
use function strtolower;
class Request
{
    /** @var SymfonyRequest */
    private $request;
    /**
     * Request constructor.
     * @param SymfonyRequest $request
     */
    private function __construct(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Request $request)
    {
        $this->request = $request;
    }
    /**
     * @return Request
     */
    public static function createFromGlobals() : \SupportPal\WhmcsIntegration\Request\Request
    {
        return new self(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Request::createFromGlobals());
    }
    /**
     * @param string $key
     * @param mixed|null $default
     * @return mixed|null
     */
    public function get(string $key, $default = null)
    {
        $value = $this->request->get($key, $default);
        if (\is_string($value)) {
            $value = \WHMCS\Input\Sanitize::decode($value);
        }
        return $value;
    }
    /**
     * @return bool
     */
    public function isPostRequest() : bool
    {
        return $this->request->isMethod('POST');
    }
    /**
     * @return array<mixed>
     */
    public function getHeaders() : array
    {
        return $this->request->headers->all();
    }
    public function getContent() : string
    {
        return (string) $this->request->getContent();
    }
    public function getRequestUri() : string
    {
        return $this->request->getRequestUri();
    }
    public function getMethod() : string
    {
        return $this->request->getMethod();
    }
    public function getClientIp() : ?string
    {
        return $this->request->getClientIp();
    }
    /**
     * @return array<mixed>
     */
    public function getBodyArray() : array
    {
        return $this->sanitizeArray($this->request->request->all());
    }
    /**
     * @return FileBag<mixed>
     */
    public function getFiles() : \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\FileBag
    {
        return $this->request->files;
    }
    /**
     * @return ServerBag<mixed>
     */
    public function getServer() : \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\ServerBag
    {
        return $this->request->server;
    }
    /**
     * Determine if the current request is asking for JSON.
     */
    public function wantsJson() : bool
    {
        $acceptable = $this->request->getAcceptableContentTypes();
        return isset($acceptable[0]) && \Illuminate\Support\Str::contains(\strtolower($acceptable[0]), ['/json', '+json']);
    }
    /**
     * @param array<mixed> $array
     * @return array<mixed>
     */
    private function sanitizeArray(array $array) : array
    {
        $sanitized = [];
        foreach ($array as $key => $value) {
            if (\is_array($value)) {
                $sanitized[$key] = $this->sanitizeArray($value);
                continue;
            }
            if (\is_string($value)) {
                $sanitized[$key] = \WHMCS\Input\Sanitize::decode($value);
                continue;
            }
            $sanitized[$key] = $value;
        }
        return $sanitized;
    }
}
