<?php

declare (strict_types=1);
namespace SupportPal\WhmcsIntegration\Service\Manager;

use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\InvalidArgumentException;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\User\User;
use SupportPal\WhmcsIntegration\Exception\InvalidApiResponse;
use SupportPal\WhmcsIntegration\Repository\User\UserCustomFieldsRepository;
use SupportPal\WhmcsIntegration\Repository\UserRepository;
use SupportPal\WhmcsIntegration\Repository\Whmcs\ClientRepository;
use SupportPal\WhmcsIntegration\Repository\Whmcs\WhmcsCustomFieldsRepository;
use SupportPal\WhmcsIntegration\Request\Request;
use SupportPal\WhmcsIntegration\Service\Converter\ModelToArrayConverter;
use WHMCS\Input\Sanitize;
use WHMCS\User\Relations\UserClient;
use function array_keys;
use function in_array;
class WhmcsSupportPalUserClientOrchestrator
{
    /** * @var ModelToArrayConverter */
    private $modelToArrayConverter;
    /** @var UserCustomFieldsRepository */
    private $userCustomFieldsRepository;
    /** @var WhmcsCustomFieldsRepository */
    private $customFieldsRepository;
    /** @var UserRepository */
    private $userRepository;
    /** @var UserManager */
    private $userManager;
    /** @var ClientRepository */
    private $clientRepository;
    /**
     * WhmcsSupportPalUserClientOrchestrator constructor.
     *
     * @param ModelToArrayConverter $modelToArrayConverter
     * @param UserCustomFieldsRepository $userCustomFieldsRepository
     * @param WhmcsCustomFieldsRepository $customFieldsRepository
     * @param UserRepository $userRepository
     * @param UserManager $userManager
     * @param ClientRepository $clientRepository
     */
    public function __construct(\SupportPal\WhmcsIntegration\Service\Converter\ModelToArrayConverter $modelToArrayConverter, \SupportPal\WhmcsIntegration\Repository\User\UserCustomFieldsRepository $userCustomFieldsRepository, \SupportPal\WhmcsIntegration\Repository\Whmcs\WhmcsCustomFieldsRepository $customFieldsRepository, \SupportPal\WhmcsIntegration\Repository\UserRepository $userRepository, \SupportPal\WhmcsIntegration\Service\Manager\UserManager $userManager, \SupportPal\WhmcsIntegration\Repository\Whmcs\ClientRepository $clientRepository)
    {
        $this->modelToArrayConverter = $modelToArrayConverter;
        $this->userCustomFieldsRepository = $userCustomFieldsRepository;
        $this->customFieldsRepository = $customFieldsRepository;
        $this->userRepository = $userRepository;
        $this->userManager = $userManager;
        $this->clientRepository = $clientRepository;
    }
    /**
     * @param mixed[] $vars
     * @throws InvalidApiResponse
     * @throws InvalidArgumentException
     */
    public function editClient(array $vars) : void
    {
        $user = $this->userRepository->getHelpdeskAccount(null, $vars['olddata']['email']);
        if (empty($user)) {
            return;
        }
        $data = $this->getClientBaseData($vars['firstname'], $vars['lastname'], $vars['email'], $vars['country'], $vars['companyname']);
        $data = $this->handleCustomFields($data, $vars);
        $this->userManager->update((new \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\User\User())->setId($user['id']), $data);
    }
    /**
     * @param string $firstName
     * @param string $lastName
     * @param string $email
     * @param string $country
     * @param string|null $companyName
     * @return mixed[]
     */
    private function getClientBaseData(string $firstName, string $lastName, string $email, string $country, ?string $companyName) : array
    {
        return ['firstname' => \WHMCS\Input\Sanitize::decode($firstName), 'lastname' => \WHMCS\Input\Sanitize::decode($lastName), 'email' => $email, 'country' => $country, 'confirmed' => 1, 'organisation' => !empty($companyName) ? \WHMCS\Input\Sanitize::decode($companyName) : null];
    }
    /**
     * @param mixed[] $vars
     * @throws InvalidApiResponse
     * @throws InvalidArgumentException
     */
    public function addClient(array $vars) : void
    {
        $data = $this->getClientBaseData($vars['firstname'], $vars['lastname'], $vars['email'], $vars['country'], $vars['companyname']);
        $data['password'] = $vars['password'];
        $data = $this->handleCustomFields($data, $vars);
        $user = $this->userRepository->getHelpdeskAccount(null, $vars['email']);
        $this->createOrUpdateUser($user, $data);
    }
    /**
     * @param string $email
     * @param string $password
     * @throws InvalidApiResponse
     * @throws InvalidArgumentException
     */
    public function updateClientPassword(string $email, string $password) : void
    {
        $user = $this->userRepository->getHelpdeskAccount(null, $email);
        if ($user === null) {
            return;
        }
        $this->userManager->update((new \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\User\User())->setId($user['id']), ['password' => $password]);
    }
    /**
     * @param mixed[] $vars
     * @throws InvalidApiResponse
     * @throws InvalidArgumentException
     */
    public function userAdd(array $vars) : void
    {
        $data = ['firstname' => \WHMCS\Input\Sanitize::decode($vars['firstname']), 'lastname' => \WHMCS\Input\Sanitize::decode($vars['lastname']), 'email' => $vars['email'], 'password' => $vars['password'], 'confirmed' => 1];
        $user = $this->userRepository->getHelpdeskAccount(null, $vars['email']);
        $this->createOrUpdateUser($user, $data);
    }
    /**
     * @param UserClient $model
     * @throws InvalidApiResponse
     * @throws InvalidArgumentException
     */
    public function onUserClientSaved(\WHMCS\User\Relations\UserClient $model) : void
    {
        if ($model->user === null) {
            return;
        }
        $parent = $this->clientRepository->getOneById((int) $model->client_id);
        if ($parent === null) {
            return;
        }
        $helpdeskUser = $this->userRepository->getHelpdeskAccount(null, $parent->email);
        if (empty($helpdeskUser['organisation_id'])) {
            return;
        }
        $data = ['firstname' => \WHMCS\Input\Sanitize::decode($model->user->first_name), 'lastname' => \WHMCS\Input\Sanitize::decode($model->user->last_name), 'organisation_id' => $helpdeskUser['organisation_id'], 'organisation_access_level' => 0, 'organisation_notifications' => 1];
        $user = $this->userRepository->getHelpdeskAccount(null, $model->user->email);
        // If user doesn't exist at SupportPal.
        if ($user === null) {
            return;
        }
        // If user is already part of an organisation, don't update.
        if (!empty($user['organisation_id'])) {
            return;
        }
        $this->userManager->update((new \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\User\User())->setId($user['id']), $data);
    }
    /**
     * @param UserClient $model
     * @throws InvalidApiResponse
     * @throws InvalidArgumentException
     */
    public function onUserClientDeleted(\WHMCS\User\Relations\UserClient $model) : void
    {
        if ($model->user === null) {
            return;
        }
        // Ensure user exists at help desk.
        $user = $this->userRepository->getHelpdeskAccount(null, $model->user->email);
        if ($user === null) {
            return;
        }
        // Get linked client account.
        $parent = $this->clientRepository->getOneById((int) $model->client_id);
        if ($parent === null) {
            return;
        }
        // Fetch help desk account for client and ensure the organisation ID matches.
        $helpdeskUser = $this->userRepository->getHelpdeskAccount(null, $parent->email);
        if ($helpdeskUser === null || $user['organisation_id'] !== $helpdeskUser['organisation_id']) {
            return;
        }
        // Remove the user from the organisation.
        $this->createOrUpdateUser($user, ['organisation_id' => 0]);
    }
    /**
     * @param mixed[] $customFieldsData
     * @param mixed[] $vars
     * @param mixed[] $data
     * @return mixed[]
     */
    private function updateWhmcsCustomFields(array $customFieldsData, array $vars, array $data)
    {
        $mapping = ['Address 1' => $vars['address1'], 'Address 2' => $vars['address2'], 'City' => $vars['city'], 'State' => $vars['state'], 'Postcode' => $vars['postcode'], 'Phone Number' => $vars['phonenumber']];
        foreach ($customFieldsData as $field) {
            if (!\in_array($field['name'], \array_keys($mapping))) {
                continue;
            }
            $data['customfield'][$field['id']] = $mapping[$field['name']];
        }
        return $data;
    }
    /**
     * @param mixed[] $whmcsFields
     * @param mixed[] $customFieldData
     * @param mixed[] $customField
     * @param mixed[] $data
     * @return mixed[]
     */
    private function addWhmcsCustomFields(array $whmcsFields, array $customFieldData, array $customField, array $data)
    {
        foreach ($whmcsFields as $field) {
            foreach ($customFieldData as $key) {
                if ($key['name'] !== $field->fieldname) {
                    continue;
                }
                if ($field->fieldtype === 'tickbox') {
                    $customField[$field->id] = $customField[$field->id] === 'on' ? 1 : 0;
                }
                $data['customfield'][$key['id']] = $customField[$field->id];
            }
        }
        return $data;
    }
    /**
     * @param mixed[] $data
     * @param mixed[] $vars
     * @return mixed[]
     * @throws InvalidApiResponse
     * @throws InvalidArgumentException
     */
    private function handleCustomFields(array $data, array $vars) : array
    {
        $customFields = $this->userCustomFieldsRepository->findBy([]);
        $customFieldData = $this->modelToArrayConverter->convertMany($customFields->getModels());
        $data['customfield'] = [];
        $data = $this->updateWhmcsCustomFields($customFieldData, $vars, $data);
        $request = \SupportPal\WhmcsIntegration\Request\Request::createFromGlobals();
        /** @var mixed[]|null $customField */
        $customField = $request->get('customfield');
        if ($customField !== null) {
            $whmcsFields = $this->customFieldsRepository->findBy('id', \array_keys($customField))->toArray();
            $data = $this->addWhmcsCustomFields($whmcsFields, $customFieldData, $customField, $data);
        }
        return $data;
    }
    /**
     * @param mixed[]|null $user
     * @param mixed[] $data
     * @throws InvalidArgumentException
     * @throws InvalidApiResponse
     */
    private function createOrUpdateUser(?array $user, array $data) : void
    {
        if (!empty($user)) {
            $this->userManager->update((new \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\User\User())->setId($user['id']), $data);
        } else {
            $this->userManager->create($data);
        }
    }
}
