<?php

declare (strict_types=1);
namespace SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient;

use SupportPal\WhmcsIntegration\Vendor\Psr\Http\Client\ClientExceptionInterface;
use SupportPal\WhmcsIntegration\Vendor\Psr\Http\Client\ClientInterface;
use SupportPal\WhmcsIntegration\Vendor\Psr\Http\Message\RequestInterface;
use SupportPal\WhmcsIntegration\Vendor\Psr\Http\Message\ResponseInterface;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\ApiClient\ApiClientAware;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\HttpResponseException;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Factory\RequestFactory;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Encoder\DecoderInterface;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Exception\NotEncodableValueException;
use function is_array;
/**
 * This class includes all the API calls
 * Class ApiClient
 * @package SupportPal\ApiClient
 */
class ApiClient
{
    use ApiClientAware;
    /** @var ClientInterface */
    private $httpClient;
    /** @var RequestFactory */
    private $requestFactory;
    /** @var DecoderInterface */
    private $decoder;
    /** @var string */
    private $formatType;
    /**
     * ApiClient constructor.
     * @param ClientInterface $httpClient
     * @param RequestFactory $requestFactory
     * @param DecoderInterface $decoder
     * @param string $formatType
     */
    public function __construct(\SupportPal\WhmcsIntegration\Vendor\Psr\Http\Client\ClientInterface $httpClient, \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Factory\RequestFactory $requestFactory, \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Encoder\DecoderInterface $decoder, string $formatType)
    {
        $this->httpClient = $httpClient;
        $this->requestFactory = $requestFactory;
        $this->decoder = $decoder;
        $this->formatType = $formatType;
    }
    /**
     * @inheritDoc
     */
    public function sendRequest(\SupportPal\WhmcsIntegration\Vendor\Psr\Http\Message\RequestInterface $request) : \SupportPal\WhmcsIntegration\Vendor\Psr\Http\Message\ResponseInterface
    {
        try {
            $response = $this->getHttpClient()->sendRequest($request);
            $this->assertRequestSuccessful($request, $response);
        } catch (\SupportPal\WhmcsIntegration\Vendor\Psr\Http\Client\ClientExceptionInterface $exception) {
            throw new \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\HttpResponseException($request, null, $exception->getMessage(), 0, $exception);
        }
        return $response;
    }
    /**
     * @inheritDoc
     */
    protected function sendDownloadRequest(\SupportPal\WhmcsIntegration\Vendor\Psr\Http\Message\RequestInterface $request) : \SupportPal\WhmcsIntegration\Vendor\Psr\Http\Message\ResponseInterface
    {
        try {
            $response = $this->getHttpClient()->sendRequest($request);
        } catch (\SupportPal\WhmcsIntegration\Vendor\Psr\Http\Client\ClientExceptionInterface $exception) {
            throw new \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\HttpResponseException($request, null, $exception->getMessage(), 0, $exception);
        }
        /**
         * response is not a file, assert a valid response to get the actual API error
         */
        if (empty($response->getHeader('Content-Disposition'))) {
            $this->assertRequestSuccessful($request, $response);
        }
        return $response;
    }
    /**
     * @inheritDoc
     * @param array<mixed> $queryParameters
     */
    protected function prepareAndSendGetRequest(string $endpoint, array $queryParameters) : \SupportPal\WhmcsIntegration\Vendor\Psr\Http\Message\ResponseInterface
    {
        $request = $this->getRequestFactory()->create('GET', $endpoint, [], [], $queryParameters);
        return $this->sendRequest($request);
    }
    /**
     * @inheritDoc
     */
    protected function assertRequestSuccessful(\SupportPal\WhmcsIntegration\Vendor\Psr\Http\Message\RequestInterface $request, \SupportPal\WhmcsIntegration\Vendor\Psr\Http\Message\ResponseInterface $response) : void
    {
        try {
            $body = $this->decoder->decode((string) $response->getBody(), $this->formatType);
        } catch (\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Exception\NotEncodableValueException $notEncodableValueException) {
            throw new \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\HttpResponseException($request, $response, $response->getReasonPhrase(), $notEncodableValueException->getCode(), $notEncodableValueException);
        }
        if ($response->getStatusCode() !== 200 || !\is_array($body) || !isset($body['status']) || $body['status'] === 'error') {
            throw new \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\HttpResponseException($request, $response, $body['message'] ?? '');
        }
    }
    /**
     * @inheritDoc
     */
    protected function getHttpClient() : \SupportPal\WhmcsIntegration\Vendor\Psr\Http\Client\ClientInterface
    {
        return $this->httpClient;
    }
    /**
     * @inheritDoc
     */
    protected function getRequestFactory() : \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Factory\RequestFactory
    {
        return $this->requestFactory;
    }
}
