<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Util;

use SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\PhpDoc\ParamTagValueNode;
use SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\PhpDoc\PhpDocTagValueNode;
use SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\PhpDoc\ReturnTagValueNode;
use SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\PhpDoc\VarTagValueNode;
use SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\ArrayShapeNode;
use SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\ArrayTypeNode;
use SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\CallableTypeNode;
use SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\CallableTypeParameterNode;
use SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\ConstTypeNode;
use SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\GenericTypeNode;
use SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\IdentifierTypeNode;
use SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\NullableTypeNode;
use SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\ThisTypeNode;
use SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\TypeNode;
use SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\UnionTypeNode;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\PhpStan\NameScope;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type;
/**
 * Transforms a php doc tag value to a {@link Type} instance.
 *
 * @author Baptiste Leduc <baptiste.leduc@gmail.com>
 *
 * @internal
 */
final class PhpStanTypeHelper
{
    /**
     * Creates a {@see Type} from a PhpDocTagValueNode type.
     *
     * @return Type[]
     */
    public function getTypes(\SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\PhpDoc\PhpDocTagValueNode $node, \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\PhpStan\NameScope $nameScope) : array
    {
        if ($node instanceof \SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\PhpDoc\ParamTagValueNode || $node instanceof \SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\PhpDoc\ReturnTagValueNode || $node instanceof \SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\PhpDoc\VarTagValueNode) {
            return $this->compressNullableType($this->extractTypes($node->type, $nameScope));
        }
        return [];
    }
    /**
     * Because PhpStan extract null as a separated type when Symfony / PHP compress it in the first available type we
     * need this method to mimic how Symfony want null types.
     *
     * @param Type[] $types
     *
     * @return Type[]
     */
    private function compressNullableType(array $types) : array
    {
        $firstTypeIndex = null;
        $nullableTypeIndex = null;
        foreach ($types as $k => $type) {
            if (null === $firstTypeIndex && \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_NULL !== $type->getBuiltinType() && !$type->isNullable()) {
                $firstTypeIndex = $k;
            }
            if (null === $nullableTypeIndex && \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_NULL === $type->getBuiltinType()) {
                $nullableTypeIndex = $k;
            }
            if (null !== $firstTypeIndex && null !== $nullableTypeIndex) {
                break;
            }
        }
        if (null !== $firstTypeIndex && null !== $nullableTypeIndex) {
            $firstType = $types[$firstTypeIndex];
            $types[$firstTypeIndex] = new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type($firstType->getBuiltinType(), \true, $firstType->getClassName(), $firstType->isCollection(), $firstType->getCollectionKeyTypes(), $firstType->getCollectionValueTypes());
            unset($types[$nullableTypeIndex]);
        }
        return \array_values($types);
    }
    /**
     * @return Type[]
     */
    private function extractTypes(\SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\TypeNode $node, \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\PhpStan\NameScope $nameScope) : array
    {
        if ($node instanceof \SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\UnionTypeNode) {
            $types = [];
            foreach ($node->types as $type) {
                if ($type instanceof \SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\ConstTypeNode) {
                    // It's safer to fall back to other extractors here, as resolving const types correctly is not easy at the moment
                    return [];
                }
                foreach ($this->extractTypes($type, $nameScope) as $subType) {
                    $types[] = $subType;
                }
            }
            return $this->compressNullableType($types);
        }
        if ($node instanceof \SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\GenericTypeNode) {
            [$mainType] = $this->extractTypes($node->type, $nameScope);
            if (\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_INT === $mainType->getBuiltinType()) {
                return [$mainType];
            }
            $collectionKeyTypes = $mainType->getCollectionKeyTypes();
            $collectionKeyValues = [];
            if (1 === \count($node->genericTypes)) {
                foreach ($this->extractTypes($node->genericTypes[0], $nameScope) as $subType) {
                    $collectionKeyValues[] = $subType;
                }
            } elseif (2 === \count($node->genericTypes)) {
                foreach ($this->extractTypes($node->genericTypes[0], $nameScope) as $keySubType) {
                    $collectionKeyTypes[] = $keySubType;
                }
                foreach ($this->extractTypes($node->genericTypes[1], $nameScope) as $valueSubType) {
                    $collectionKeyValues[] = $valueSubType;
                }
            }
            return [new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type($mainType->getBuiltinType(), $mainType->isNullable(), $mainType->getClassName(), \true, $collectionKeyTypes, $collectionKeyValues)];
        }
        if ($node instanceof \SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\ArrayShapeNode) {
            return [new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_ARRAY, \false, null, \true)];
        }
        if ($node instanceof \SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\ArrayTypeNode) {
            return [new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_ARRAY, \false, null, \true, [new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_INT)], $this->extractTypes($node->type, $nameScope))];
        }
        if ($node instanceof \SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\CallableTypeNode || $node instanceof \SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\CallableTypeParameterNode) {
            return [new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_CALLABLE)];
        }
        if ($node instanceof \SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\NullableTypeNode) {
            $subTypes = $this->extractTypes($node->type, $nameScope);
            if (\count($subTypes) > 1) {
                $subTypes[] = new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_NULL);
                return $subTypes;
            }
            return [new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type($subTypes[0]->getBuiltinType(), \true, $subTypes[0]->getClassName(), $subTypes[0]->isCollection(), $subTypes[0]->getCollectionKeyTypes(), $subTypes[0]->getCollectionValueTypes())];
        }
        if ($node instanceof \SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\ThisTypeNode) {
            return [new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_OBJECT, \false, $nameScope->resolveRootClass())];
        }
        if ($node instanceof \SupportPal\WhmcsIntegration\Vendor\PHPStan\PhpDocParser\Ast\Type\IdentifierTypeNode) {
            if (\in_array($node->name, \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::$builtinTypes)) {
                return [new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type($node->name, \false, null, \in_array($node->name, \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::$builtinCollectionTypes))];
            }
            switch ($node->name) {
                case 'integer':
                    return [new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_INT)];
                case 'list':
                case 'non-empty-list':
                    return [new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_ARRAY, \false, null, \true, new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_INT))];
                case 'non-empty-array':
                    return [new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_ARRAY, \false, null, \true)];
                case 'mixed':
                    return [];
                // mixed seems to be ignored in all other extractors
                case 'parent':
                    return [new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_OBJECT, \false, $node->name)];
                case 'static':
                case 'self':
                    return [new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_OBJECT, \false, $nameScope->resolveRootClass())];
                case 'void':
                    return [new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_NULL)];
            }
            return [new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_OBJECT, \false, $nameScope->resolveStringName($node->name))];
        }
        return [];
    }
}
