<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Normalizer;

use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Exception\InvalidArgumentException;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Exception\NotNormalizableValueException;
/**
 * Normalizes a {@see \BackedEnum} enumeration to a string or an integer.
 *
 * @author Alexandre Daubois <alex.daubois@gmail.com>
 */
final class BackedEnumNormalizer implements \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Normalizer\NormalizerInterface, \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Normalizer\DenormalizerInterface, \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Normalizer\CacheableSupportsMethodInterface
{
    /**
     * {@inheritdoc}
     *
     * @return int|string
     *
     * @throws InvalidArgumentException
     */
    public function normalize($object, string $format = null, array $context = [])
    {
        if (!$object instanceof \BackedEnum) {
            throw new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Exception\InvalidArgumentException('The data must belong to a backed enumeration.');
        }
        return $object->value;
    }
    /**
     * {@inheritdoc}
     */
    public function supportsNormalization($data, string $format = null) : bool
    {
        return $data instanceof \BackedEnum;
    }
    /**
     * {@inheritdoc}
     *
     * @throws NotNormalizableValueException
     */
    public function denormalize($data, string $type, string $format = null, array $context = [])
    {
        if (!\is_subclass_of($type, \BackedEnum::class)) {
            throw new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Exception\InvalidArgumentException('The data must belong to a backed enumeration.');
        }
        if (!\is_int($data) && !\is_string($data)) {
            throw \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Exception\NotNormalizableValueException::createForUnexpectedDataType('The data is neither an integer nor a string, you should pass an integer or a string that can be parsed as an enumeration case of type ' . $type . '.', $data, [\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_INT, \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_STRING], $context['deserialization_path'] ?? null, \true);
        }
        try {
            return $type::from($data);
        } catch (\ValueError $e) {
            throw new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Exception\InvalidArgumentException('The data must belong to a backed enumeration of type ' . $type);
        }
    }
    /**
     * {@inheritdoc}
     */
    public function supportsDenormalization($data, string $type, string $format = null) : bool
    {
        return \is_subclass_of($type, \BackedEnum::class);
    }
    /**
     * {@inheritdoc}
     */
    public function hasCacheableSupportsMethod() : bool
    {
        return \true;
    }
}
