<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Normalizer;

use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Exception\InvalidArgumentException;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Exception\NotNormalizableValueException;
/**
 * Normalizes a {@see \DateTimeZone} object to a timezone string.
 *
 * @author Jérôme Desjardins <jewome62@gmail.com>
 */
class DateTimeZoneNormalizer implements \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Normalizer\NormalizerInterface, \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Normalizer\DenormalizerInterface, \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Normalizer\CacheableSupportsMethodInterface
{
    /**
     * {@inheritdoc}
     *
     * @return string
     *
     * @throws InvalidArgumentException
     */
    public function normalize($object, string $format = null, array $context = [])
    {
        if (!$object instanceof \DateTimeZone) {
            throw new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Exception\InvalidArgumentException('The object must be an instance of "\\DateTimeZone".');
        }
        return $object->getName();
    }
    /**
     * {@inheritdoc}
     */
    public function supportsNormalization($data, string $format = null)
    {
        return $data instanceof \DateTimeZone;
    }
    /**
     * {@inheritdoc}
     *
     * @return \DateTimeZone
     *
     * @throws NotNormalizableValueException
     */
    public function denormalize($data, string $type, string $format = null, array $context = [])
    {
        if ('' === $data || null === $data) {
            throw \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Exception\NotNormalizableValueException::createForUnexpectedDataType('The data is either an empty string or null, you should pass a string that can be parsed as a DateTimeZone.', $data, [\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_STRING], $context['deserialization_path'] ?? null, \true);
        }
        try {
            return new \DateTimeZone($data);
        } catch (\Exception $e) {
            throw \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Serializer\Exception\NotNormalizableValueException::createForUnexpectedDataType($e->getMessage(), $data, [\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\PropertyInfo\Type::BUILTIN_TYPE_STRING], $context['deserialization_path'] ?? null, \true, $e->getCode(), $e);
        }
    }
    /**
     * {@inheritdoc}
     */
    public function supportsDenormalization($data, string $type, string $format = null)
    {
        return \DateTimeZone::class === $type;
    }
    /**
     * {@inheritdoc}
     */
    public function hasCacheableSupportsMethod() : bool
    {
        return __CLASS__ === static::class;
    }
}
