<?php

declare (strict_types=1);
namespace SupportPal\WhmcsIntegration\Controller\SelfService;

use Closure;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\Exception;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\InvalidArgumentException;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\RuntimeException;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Collection\Collection;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\SelfService\Article;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\SelfService\Category;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\SelfService\Type;
use SupportPal\WhmcsIntegration\Exception\InvalidApiResponse;
use SupportPal\WhmcsIntegration\Factory\ClientAreaFactory;
use SupportPal\WhmcsIntegration\Helper\AuthenticatedUserHelper;
use SupportPal\WhmcsIntegration\Helper\CaptchaHelper;
use SupportPal\WhmcsIntegration\Helper\DateTimeFormatter;
use SupportPal\WhmcsIntegration\Helper\Paginator;
use SupportPal\WhmcsIntegration\Helper\StringHelper;
use SupportPal\WhmcsIntegration\Log\ExceptionLogger;
use SupportPal\WhmcsIntegration\Repository\Core\CoreSettingsRepository;
use SupportPal\WhmcsIntegration\Repository\SelfService\ArticleRepository;
use SupportPal\WhmcsIntegration\Repository\SelfService\ArticleTypeRepository;
use SupportPal\WhmcsIntegration\Repository\SelfService\CategoryRepository;
use SupportPal\WhmcsIntegration\Repository\SelfService\CommentRepository;
use SupportPal\WhmcsIntegration\Repository\SelfService\SelfServiceSettingsRepository;
use SupportPal\WhmcsIntegration\Repository\SelfService\TagRepository;
use SupportPal\WhmcsIntegration\Repository\UserRepository;
use SupportPal\WhmcsIntegration\Repository\Whmcs\ModSupportPalRepository;
use SupportPal\WhmcsIntegration\Request\Request;
use SupportPal\WhmcsIntegration\Service\Converter\ModelToArrayConverter;
use SupportPal\WhmcsIntegration\Service\Manager\Locale\LocaleManager;
use SupportPal\WhmcsIntegration\Service\Manager\SelfService\CommentManager;
use WHMCS\Application\Support\Facades\Lang;
use WHMCS\Application\Support\Facades\Menu;
use WHMCS\ClientArea;
use function array_filter;
use function count;
use function htmlentities;
use function is_numeric;
use function rtrim;
abstract class BaseSelfServiceController
{
    protected const TYPE = '';
    protected const DEFAULT_PAGE_TITLE = '';
    protected const ARTICLES_PER_PAGE = 10;
    /** @var DateTimeFormatter */
    protected $dateTimeFormatter;
    /** @var ModelToArrayConverter */
    protected $modelToArrayConverter;
    /** @var Request */
    protected $request;
    /** @var ArticleRepository */
    protected $articleRepository;
    /** @var SelfServiceSettingsRepository */
    protected $settingsRepository;
    /** @var CategoryRepository */
    protected $categoryRepository;
    /** @var TagRepository */
    protected $tagRepository;
    /** @var string[] */
    protected $locale;
    /** @var CoreSettingsRepository */
    protected $coreSettingsRepository;
    /** @var CommentRepository */
    protected $commentRepository;
    /** @var CommentManager */
    protected $commentManager;
    /** @var CaptchaHelper */
    protected $captchaHelper;
    /** @var StringHelper */
    protected $stringHelper;
    /** @var AuthenticatedUserHelper */
    protected $authenticatedUserHelper;
    /** @var UserRepository */
    protected $userRepository;
    /** @var ModSupportPalRepository */
    protected $modSupportPalRepository;
    /** @var ArticleTypeRepository */
    protected $articleTypeRepository;
    /** @var int|null */
    protected $typeId;
    /** @var ClientAreaFactory */
    private $clientAreaFactory;
    /** @var Paginator */
    protected $paginator;
    /** @var ExceptionLogger */
    protected $exceptionLogger;
    public function __construct(\SupportPal\WhmcsIntegration\Helper\DateTimeFormatter $dateTimeFormatter, \SupportPal\WhmcsIntegration\Service\Converter\ModelToArrayConverter $modelToArrayConverter, \SupportPal\WhmcsIntegration\Repository\SelfService\ArticleRepository $articleRepository, \SupportPal\WhmcsIntegration\Repository\SelfService\SelfServiceSettingsRepository $settingsRepository, \SupportPal\WhmcsIntegration\Repository\SelfService\CategoryRepository $categoryRepository, \SupportPal\WhmcsIntegration\Repository\SelfService\TagRepository $tagRepository, \SupportPal\WhmcsIntegration\Repository\Core\CoreSettingsRepository $coreSettingsRepository, \SupportPal\WhmcsIntegration\Repository\SelfService\CommentRepository $commentRepository, \SupportPal\WhmcsIntegration\Service\Manager\SelfService\CommentManager $commentManager, \SupportPal\WhmcsIntegration\Helper\CaptchaHelper $captchaHelper, \SupportPal\WhmcsIntegration\Helper\StringHelper $stringHelper, \SupportPal\WhmcsIntegration\Helper\AuthenticatedUserHelper $authenticatedUserHelper, \SupportPal\WhmcsIntegration\Repository\UserRepository $userRepository, \SupportPal\WhmcsIntegration\Repository\Whmcs\ModSupportPalRepository $modSupportPalRepository, \SupportPal\WhmcsIntegration\Repository\SelfService\ArticleTypeRepository $articleTypeRepository, \SupportPal\WhmcsIntegration\Service\Manager\Locale\LocaleManager $localeManager, \SupportPal\WhmcsIntegration\Factory\ClientAreaFactory $clientAreaFactory, \SupportPal\WhmcsIntegration\Helper\Paginator $paginator, \SupportPal\WhmcsIntegration\Log\ExceptionLogger $exceptionLogger)
    {
        $this->dateTimeFormatter = $dateTimeFormatter;
        $this->modelToArrayConverter = $modelToArrayConverter;
        $this->articleRepository = $articleRepository;
        $this->settingsRepository = $settingsRepository;
        $this->categoryRepository = $categoryRepository;
        $this->tagRepository = $tagRepository;
        $this->coreSettingsRepository = $coreSettingsRepository;
        $this->commentRepository = $commentRepository;
        $this->commentManager = $commentManager;
        $this->captchaHelper = $captchaHelper;
        $this->stringHelper = $stringHelper;
        $this->authenticatedUserHelper = $authenticatedUserHelper;
        $this->userRepository = $userRepository;
        $this->modSupportPalRepository = $modSupportPalRepository;
        $this->articleTypeRepository = $articleTypeRepository;
        $this->clientAreaFactory = $clientAreaFactory;
        $this->locale = $localeManager->getLocale(\WHMCS\Application\Support\Facades\Lang::trans('locale'));
        $this->paginator = $paginator->setPerPage(static::ARTICLES_PER_PAGE);
        $this->exceptionLogger = $exceptionLogger;
    }
    public function getTypeName() : string
    {
        return static::TYPE;
    }
    /**
     * @param int $id
     * @param Request $request
     * @return void
     */
    public function showOneById(int $id, \SupportPal\WhmcsIntegration\Request\Request $request) : void
    {
        $clientArea = $this->prologue($this->getArticleTemplatePath());
        try {
            $this->setTypeId();
            $this->showOne($clientArea, $id, $request);
        } catch (\SupportPal\WhmcsIntegration\Exception\InvalidApiResponse|\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\Exception|\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\RuntimeException $exception) {
            $this->exceptionLogger->log($exception, __FUNCTION__);
            $this->showNotFoundPage($clientArea, 'articles');
        }
    }
    /**
     * This route shows all articles
     */
    public function showAll(\SupportPal\WhmcsIntegration\Request\Request $request) : void
    {
        $clientArea = $this->prologue($this->getAllTemplatePath());
        try {
            $this->setTypeId();
            $this->showAllArticles($clientArea);
        } catch (\SupportPal\WhmcsIntegration\Exception\InvalidApiResponse|\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\Exception|\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\RuntimeException $exception) {
            $this->exceptionLogger->log($exception, __FUNCTION__);
            $this->showNotFoundPage($clientArea, 'popularArticles');
        }
    }
    /**
     * This route shows articles list filtered by tag, category, or search term
     *
     * @param string|null $search
     * @param int|null $tagId
     * @param int|null $categoryId
     * @param int $currentPage
     */
    public function showFilteredArticles(?string $search, ?int $tagId, ?int $categoryId, int $currentPage) : void
    {
        $clientArea = $this->prologue($this->getFilteredTemplate());
        try {
            $this->setTypeId();
            $this->showFiltered($clientArea, $categoryId, $currentPage, $search, $tagId);
        } catch (\SupportPal\WhmcsIntegration\Exception\InvalidApiResponse|\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\Exception|\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\RuntimeException $exception) {
            $this->exceptionLogger->log($exception, __FUNCTION__);
            $this->showNotFoundPage($clientArea, 'articles');
        }
    }
    /**
     * @param mixed[] $selfserviceSettings
     * @param ClientArea $ca
     * @param Request $request
     * @param int $id
     * @param int|null $typeId
     * @param mixed[]|null $helpdeskUser
     * @throws InvalidArgumentException
     */
    protected function handlePostComment(array $selfserviceSettings, \WHMCS\ClientArea $ca, \SupportPal\WhmcsIntegration\Request\Request $request, int $id, ?int $typeId, ?array $helpdeskUser) : void
    {
        $isValid = \true;
        // If captcha is set, check to see the answer is correct
        if ($selfserviceSettings['comment_captcha'] === '2' || !$ca->isLoggedIn() && $selfserviceSettings['comment_captcha'] === '1') {
            $isValid = $this->captchaHelper->verify($request->get('captcha'));
        }
        // Should the comment be submitted?
        if ($isValid) {
            // Add article ID and type ID to data
            $comment = $request->getBodyArray();
            $comment['comment_item_id'] = $id;
            $comment['comment_type_id'] = $typeId;
            try {
                $commentStatus = $this->commentManager->createComment($comment, $selfserviceSettings, $helpdeskUser)->getStatus();
            } catch (\SupportPal\WhmcsIntegration\Exception\InvalidApiResponse $invalidApiResponse) {
                $commentStatus = -2;
            }
        } else {
            // Captcha was wrong, show an error
            $commentStatus = -1;
        }
        if ($commentStatus === -1 || $commentStatus === -2) {
            // Restore the form values
            $commentParentId = $request->get('comment_parent_id', null);
            if ($commentParentId !== null && \is_numeric($commentParentId)) {
                $ca->assign('comment_parent_id', $commentParentId);
            }
            $commentMessage = $request->get('comment_message', null);
            $ca->assign('orig_comment_message', $commentMessage);
            $ca->assign('orig_comment_email_author', $request->get('comment_email_author', null));
        }
        $commentName = $request->get('comment_name', null);
        // Send to view
        $ca->assign('orig_comment_name', $commentName);
        $ca->assign('comment_status', $commentStatus);
    }
    /**
     * @return Closure
     */
    protected function formatPublishedAt() : \Closure
    {
        $coreSettings = $this->coreSettingsRepository->get();
        return function (array $input) use($coreSettings) {
            $input['published_at'] = $this->dateTimeFormatter->format((string) $coreSettings->get('date_format', 'd/m/Y'), (string) $coreSettings->get('time_format', 'g:i A'), $input['published_at']);
            return $input;
        };
    }
    /**
     * @param int|null $categoryId
     * @return Closure
     */
    protected function belongsPublicCategory(?int $categoryId) : \Closure
    {
        return function (\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\SelfService\Article $article) use($categoryId) {
            foreach ((array) $article->getCategories() as $category) {
                if ($categoryId !== null && $category->getId() !== $categoryId) {
                    continue;
                }
                if ($category->getPublic() && $category->getParentPublic()) {
                    return \true;
                }
            }
            return \false;
        };
    }
    /**
     * @return Closure
     */
    protected function belongsToAPublicCategory() : \Closure
    {
        return $this->belongsPublicCategory(null);
    }
    /**
     * @param string $templatePath
     * @return ClientArea
     */
    protected function prologue(string $templatePath) : \WHMCS\ClientArea
    {
        $clientArea = $this->clientAreaFactory->create();
        $clientArea->initPage();
        $clientArea->addToBreadCrumb('index.php', \WHMCS\Application\Support\Facades\Lang::trans('globalsystemname'));
        $clientArea->addToBreadCrumb(static::TYPE . '.php', \WHMCS\Application\Support\Facades\Lang::trans(static::DEFAULT_PAGE_TITLE));
        $clientArea->setTemplate($templatePath);
        return $clientArea;
    }
    /**
     * @param mixed[] $data
     */
    protected function epilogue(\WHMCS\ClientArea $clientArea, array $data, string $pageTitle) : void
    {
        $clientArea->setPageTitle($pageTitle);
        $clientArea->assign('data', $data);
        $this->showSideMenu($data);
        $baseUrl = $this->modSupportPalRepository->getDefaultBaseUrl();
        $clientArea->assign('supportpalUrl', $baseUrl !== null ? \rtrim($baseUrl, '/') . '/' : '');
        $clientArea->assign('LANG2', $this->locale);
        $clientArea->output();
    }
    /**
     * @param mixed[] $categoriesData
     * @param ClientArea $clientArea
     * @return mixed[]
     * @throws Exception
     */
    protected function getArticlesForCategories(array $categoriesData, \WHMCS\ClientArea $clientArea) : array
    {
        $filterData = ['include_subcategories' => 1, 'published' => 1, 'protected' => $clientArea->isLoggedIn() ? null : 0, 'limit' => 3, 'order_column' => 'random'];
        foreach ($categoriesData as $key => $category) {
            $articles = $this->articleRepository->findByTypeAndCategory((int) $this->getTypeId(), $category['id'], $filterData);
            $categoriesData[$key]['articles'] = $this->modelToArrayConverter->convertMany($articles->getModels());
            $categoriesData[$key]['count'] = $articles->getCount();
        }
        return $categoriesData;
    }
    /**
     * @param mixed[] $articleData
     */
    protected function showSideMenu(array $articleData) : void
    {
        if (!isset($articleData['categories'])) {
            return;
        }
        // Show the sidebar with announcements active
        $secondarySidebar = \WHMCS\Application\Support\Facades\Menu::secondarySidebar('announcementList');
        // So long as we have one category
        if (!\count($articleData['categories'])) {
            return;
        }
        // Add a new menu above for categories
        $secondarySidebar->addChild('categories', array('label' => \WHMCS\Application\Support\Facades\Lang::trans('knowledgebasecategories'), 'uri' => '#', 'icon' => 'fa-folder-open'));
        // Retrieve the panel we just created
        $menuPanel = $secondarySidebar->getChild('categories');
        if ($menuPanel === null) {
            return;
        }
        // Add each category link
        foreach ($articleData['categories'] as $category) {
            $menuPanel->addChild('category-' . \htmlentities($category['name']), ['uri' => static::TYPE . '.php?category=' . $category['id'], 'label' => ($category['count'] > 0 ? '<span class="badge pull-right float-right">' . $category['count'] . '</span>' : '') . \htmlentities($category['name'])]);
        }
    }
    /**
     * @return int|null
     */
    protected function getTypeId() : ?int
    {
        return $this->typeId;
    }
    /**
     * @return string
     */
    protected abstract function getArticleTemplatePath() : string;
    /**
     * @return string
     */
    protected abstract function getAllTemplatePath() : string;
    /**
     * @return string
     */
    protected abstract function getFilteredTemplate() : string;
    /**
     * @param Article[] $articleData
     * @param mixed[] $categoriesData
     * @param string $dataKey
     * @return mixed[]
     */
    protected function createFinalDataArray(array $articleData, array $categoriesData, string $dataKey) : array
    {
        $articleData = $this->modelToArrayConverter->convertManyAndApply($articleData, $this->formatPublishedAt());
        $articleData = ['categories' => $categoriesData, $dataKey => $articleData];
        return $articleData;
    }
    /**
     * Get the type ID defined in the addon settings.
     *
     * @return int|null
     */
    protected function getDefinedTypeId() : ?int
    {
        return $this->modSupportPalRepository->getInt(static::TYPE . '_id');
    }
    /**
     * Get the type based on it's name i.e. look for a type called "Downloads" within the brand.
     *
     * @return int|null
     * @throws InvalidArgumentException
     */
    protected function getDefaultTypeId() : ?int
    {
        $type = $this->articleTypeRepository->findOneByName(static::TYPE);
        if ($type !== null) {
            return $type->getId();
        }
        return null;
    }
    /**
     * @param ClientArea $clientArea
     * @param string $dataKey
     */
    protected function showNotFoundPage(\WHMCS\ClientArea $clientArea, string $dataKey) : void
    {
        $this->epilogue($clientArea, ['categories' => [], $dataKey => []], \WHMCS\Application\Support\Facades\Lang::trans(static::DEFAULT_PAGE_TITLE));
    }
    /**
     * @param ClientArea $clientArea
     * @param int $id
     * @param Request $request
     * @throws InvalidApiResponse
     * @throws InvalidArgumentException
     */
    private function showOne(\WHMCS\ClientArea $clientArea, int $id, \SupportPal\WhmcsIntegration\Request\Request $request) : void
    {
        if ($this->typeId === null) {
            $this->showNotFoundPage($clientArea, 'articles');
            return;
        }
        $filterData = ['type_id' => $this->getTypeId(), 'increment_views' => 1];
        $article = $this->articleRepository->find((int) $id, $filterData);
        $belongsToTypeId = \array_filter((array) $article->getTypes(), function (\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\SelfService\Type $type) {
            return $type->getId() === $this->getTypeId();
        });
        if (empty($belongsToTypeId)) {
            $this->showNotFoundPage($clientArea, 'articles');
            return;
        }
        $pageTitle = $article->getTitle();
        $clientArea->addToBreadCrumb('', \htmlentities($pageTitle));
        $article = $this->modelToArrayConverter->convertOneAndApply($article, $this->formatPublishedAt());
        $articleData = ['article' => $article];
        $selfServiceSettings = $this->settingsRepository->get();
        $selfServiceSettingsArray = $selfServiceSettings->all();
        if (!empty($selfServiceSettingsArray)) {
            $clientArea->assign('settings', $selfServiceSettingsArray);
        }
        $clientArea->assign('userLoggedIn', $clientArea->isLoggedIn());
        $user = $this->authenticatedUserHelper->getAuthenticatedUser();
        $helpdeskUser = isset($user) ? $this->userRepository->getHelpdeskAccount(null, $user->email) : null;
        $clientArea->assign('helpdeskUser', $helpdeskUser);
        if ($request->get('comment_message', null) !== null) {
            $this->handlePostComment($selfServiceSettings->all(), $clientArea, $request, $id, $this->getTypeId(), $helpdeskUser);
        }
        $shouldBuildCaptcha = $selfServiceSettings->get('comment_captcha') === '2' || !$clientArea->isLoggedIn() && $selfServiceSettings->get('comment_captcha') === '1';
        /**  Get a captcha in case they decide to post a comment */
        if ($shouldBuildCaptcha) {
            $clientArea->assign('recaptcha', $this->captchaHelper->buildHtml());
        }
        $data = ['article_id' => $id, 'type_id' => $this->getTypeId(), 'status' => 1, 'order_direction' => 'asc'];
        $articleData['comments'] = $this->commentRepository->getCommentTree($data, $this->coreSettingsRepository->get());
        $this->epilogue($clientArea, $articleData, $pageTitle);
    }
    /**
     * @param ClientArea $clientArea
     * @throws InvalidApiResponse
     * @throws InvalidArgumentException
     */
    private function showAllArticles(\WHMCS\ClientArea $clientArea) : void
    {
        if ($this->typeId === null) {
            $this->showNotFoundPage($clientArea, 'popularArticles');
            return;
        }
        $data = ['type_id' => $this->getTypeId(), 'parent_id' => 0, 'public' => 1, 'order_column' => 'name'];
        $categoriesData = $this->getArticlesForCategories($this->modelToArrayConverter->convertMany($this->categoryRepository->findBy($data)->getModels()), $clientArea);
        $data = ['type_id' => $this->getTypeId(), 'include_subcategories' => 1, 'published' => 1, 'protected' => $clientArea->isLoggedIn() ? null : 0, 'limit' => 10, 'order_column' => 'views', 'order_direction' => 'desc'];
        $mostPopularArticles = $this->articleRepository->findBy($data);
        $articleData = [];
        /** @var Article $value */
        foreach ($mostPopularArticles->getModels() as $value) {
            // Only want 5
            if (\count($articleData) >= 5) {
                break;
            }
            // Check if any of the categories are public
            $public = \false;
            /** @var Category $category */
            foreach ((array) $value->getCategories() as $category) {
                if ($category->getPublic() && $category->getParentPublic()) {
                    $public = \true;
                    break;
                }
            }
            if (!$public) {
                continue;
            }
            $articleData[] = $value;
        }
        $articleData = $this->createFinalDataArray($articleData, $categoriesData, 'popularArticles');
        $this->epilogue($clientArea, $articleData, \WHMCS\Application\Support\Facades\Lang::trans(static::DEFAULT_PAGE_TITLE));
    }
    /**
     * @param ClientArea $clientArea
     * @param int|null $categoryId
     * @param int $currentPage
     * @param string|null $search
     * @param int|null $tagId
     * @throws InvalidApiResponse
     * @throws InvalidArgumentException
     */
    private function showFiltered(\WHMCS\ClientArea $clientArea, ?int $categoryId, int $currentPage, ?string $search, ?int $tagId) : void
    {
        if ($this->typeId === null) {
            $this->showNotFoundPage($clientArea, 'articles');
            return;
        }
        /**
         * get all subcategories, with their articles as array.
         * [name => '', ..., articles=> [...]], [name => '', ..., articles=> [...]], ...[]
         */
        $subcategoryData = [];
        if ($categoryId !== null) {
            $category = $this->categoryRepository->find($categoryId);
            if ($category->getTypeId() === $this->getTypeId() && $category->getPublic() && $category->getParentPublic()) {
                $pageTitle = $category->getName();
                /**
                 * get children categories
                 */
                $data = ['type_id' => $this->getTypeId(), 'parent_id' => $categoryId, 'public' => 1, 'order_column' => 'name'];
                $subcategoryData = $this->getArticlesForCategories($this->modelToArrayConverter->convertMany($this->categoryRepository->findBy($data)->getModels()), $clientArea);
            }
        }
        /**
         * get first self::ARTICLES_PER_PAGE either using search term or (tag and category, sorted by views) combination
         */
        $articleFilters = ['type_id' => $this->getTypeId(), 'published' => 1, 'protected' => $clientArea->isLoggedIn() ? null : 0, 'start' => $this->paginator->start($currentPage), 'limit' => $this->paginator->limit()];
        if ($search !== null) {
            $articlesCollection = $this->getArticlesByTerm($clientArea, $search, $articleFilters);
        } else {
            $articleFilters['order_column'] = 'views';
            $articleFilters['order_direction'] = 'desc';
            $articlesCollection = $this->getArticlesByTagAndCategory($categoryId, $articleFilters, $tagId);
        }
        /**
         * filter articles that have public categories, or parent categories
         */
        /** @var Article[] $filteredArticles */
        $filteredArticles = $articlesCollection->filter($this->belongsPublicCategory($categoryId))->getModels();
        $articleData = $this->createFinalDataArray($filteredArticles, $subcategoryData, 'articles');
        if ($tagId !== null) {
            $pageTitle = $this->setTag($tagId, $clientArea);
        }
        $this->setupFilteredData($articlesCollection, $clientArea, $currentPage, $articleData, $pageTitle ?? \WHMCS\Application\Support\Facades\Lang::trans(static::DEFAULT_PAGE_TITLE));
    }
    /**
     * @param int $tagId
     * @param ClientArea $clientArea
     * @return string
     * @throws InvalidApiResponse
     */
    protected function setTag(int $tagId, \WHMCS\ClientArea $clientArea) : string
    {
        $tag = $this->tagRepository->find($tagId);
        $pageTitle = $tag->getName();
        $clientArea->addToBreadCrumb('', \htmlentities($pageTitle));
        $clientArea->assign('tag', $this->stringHelper->getUrlSlug($pageTitle));
        return $pageTitle;
    }
    /**
     * @param Collection $articlesCollection
     * @param ClientArea $clientArea
     * @param int $currentPage
     * @param mixed[] $articleData
     * @param string $pageTitle
     */
    protected function setupFilteredData(\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Collection\Collection $articlesCollection, \WHMCS\ClientArea $clientArea, int $currentPage, array $articleData, string $pageTitle) : void
    {
        /**
         * setup pagination
         */
        $clientArea->assign('totalPages', $this->paginator->total($articlesCollection->getCount()));
        $clientArea->assign('currentPage', $currentPage);
        $this->epilogue($clientArea, $articleData, $pageTitle);
    }
    /**
     * @param ClientArea $clientArea
     * @param string $search
     * @param mixed[] $data
     * @return Collection
     * @throws Exception
     */
    protected function getArticlesByTerm(\WHMCS\ClientArea $clientArea, string $search, array $data) : \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Collection\Collection
    {
        $clientArea->assign('search', $search);
        $data['term'] = $search;
        return $this->articleRepository->findByTerm((string) $search, $data);
    }
    /**
     * @param int|null $categoryId
     * @param mixed[] $data
     * @param int|null $tagId
     * @return Collection
     * @throws Exception
     */
    protected function getArticlesByTagAndCategory(?int $categoryId, array $data, ?int $tagId) : \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Collection\Collection
    {
        $data['category_id'] = $categoryId;
        $data['tag_id'] = $tagId;
        return $this->articleRepository->findBy($data);
    }
    protected function setTypeId() : void
    {
        $this->typeId = $this->getDefinedTypeId();
        if ($this->typeId !== null) {
            return;
        }
        $this->typeId = $this->getDefaultTypeId();
    }
}
