<?php

declare (strict_types=1);
namespace SupportPal\WhmcsIntegration\Controller;

use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\InvalidArgumentException;
use SupportPal\WhmcsIntegration\Exception\ValidationException;
use SupportPal\WhmcsIntegration\Factory\CurrentUserFactory;
use SupportPal\WhmcsIntegration\Factory\ResponseFactory;
use SupportPal\WhmcsIntegration\Helper\CsrfTokenHelper;
use SupportPal\WhmcsIntegration\Request\Request;
use SupportPal\WhmcsIntegration\Validator\ArticleTypeIdBelongsToBrandValidator;
use SupportPal\WhmcsIntegration\Validator\AuthenticationValidator;
class ValidateIntegrationSettingsController
{
    /** @var AuthenticationValidator */
    private $authenticationValidator;
    /** @var ArticleTypeIdBelongsToBrandValidator */
    private $typeIdBelongsToBrandValidator;
    /** @var ResponseFactory */
    private $responseFactory;
    /** @var CsrfTokenHelper */
    private $csrfTokenHelper;
    /** @var CurrentUserFactory */
    private $currentUserFactory;
    public function __construct(\SupportPal\WhmcsIntegration\Validator\AuthenticationValidator $authenticationValidator, \SupportPal\WhmcsIntegration\Validator\ArticleTypeIdBelongsToBrandValidator $typeIdBelongsToBrandValidator, \SupportPal\WhmcsIntegration\Factory\ResponseFactory $responseFactory, \SupportPal\WhmcsIntegration\Helper\CsrfTokenHelper $csrfTokenHelper, \SupportPal\WhmcsIntegration\Factory\CurrentUserFactory $currentUserFactory)
    {
        $this->authenticationValidator = $authenticationValidator;
        $this->typeIdBelongsToBrandValidator = $typeIdBelongsToBrandValidator;
        $this->responseFactory = $responseFactory;
        $this->csrfTokenHelper = $csrfTokenHelper;
        $this->currentUserFactory = $currentUserFactory;
    }
    /**
     * @param Request $request
     * @param int[] $typeIds
     * @param int $brandId
     * @return void
     * @throws InvalidArgumentException
     */
    public function validateIdsBelongsToBrand(\SupportPal\WhmcsIntegration\Request\Request $request, array $typeIds, int $brandId) : void
    {
        if (!$this->checkCsrfToken($request) || !$this->checkAdmin()) {
            return;
        }
        try {
            $this->typeIdBelongsToBrandValidator->validateMany($typeIds, $brandId);
        } catch (\SupportPal\WhmcsIntegration\Exception\ValidationException $validationException) {
            $this->sendErrorResponse($validationException->getMessage());
            return;
        }
        $this->responseFactory->createJsonResponse(['status' => 'success', 'message' => null])->send();
    }
    /**
     * @param Request $request
     * @param int $brandId
     */
    public function validateAuthentication(\SupportPal\WhmcsIntegration\Request\Request $request, int $brandId) : void
    {
        if (!$this->checkCsrfToken($request) || !$this->checkAdmin()) {
            return;
        }
        try {
            $this->authenticationValidator->validate($brandId);
        } catch (\SupportPal\WhmcsIntegration\Exception\ValidationException $validationException) {
            $this->sendErrorResponse($validationException->getMessage());
            return;
        }
        $this->responseFactory->createJsonResponse(['status' => 'success', 'message' => null])->send();
    }
    /**
     * @param Request $request
     * @return bool
     */
    private function checkCsrfToken(\SupportPal\WhmcsIntegration\Request\Request $request) : bool
    {
        if (!$this->csrfTokenHelper->validateToken($request->get('validate_token'))) {
            $this->sendErrorResponse('CSRF token is invalid.');
            return \false;
        }
        return \true;
    }
    /**
     * @return bool
     */
    private function checkAdmin() : bool
    {
        if (!$this->currentUserFactory->create()->isAuthenticatedAdmin()) {
            $this->sendErrorResponse('Not authenticated as an admin.');
            return \false;
        }
        return \true;
    }
    /**
     * @param string $message
     */
    private function sendErrorResponse(string $message) : void
    {
        $this->responseFactory->createJsonResponse(['status' => 'error', 'message' => $message, 'data' => null])->send();
    }
}
