<?php

declare (strict_types=1);
namespace SupportPal\WhmcsIntegration\DI;

use Exception;
use SupportPal\WhmcsIntegration\SupportPalApiClientWrapper;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Config\FileLocator;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\DependencyInjection\ContainerBuilder;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\DependencyInjection\ContainerInterface;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\DependencyInjection\Dumper\PhpDumper;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\DependencyInjection\Loader\YamlFileLoader;
use function file_exists;
use function file_put_contents;
use function unlink;
/**
 * Class Container
 * @package SupportPal\WhmcsIntegration\DI
 */
class Container
{
    private const CONTAINER_CLASS_NAME = 'SupportPalIntegrationContainer';
    private const CACHED_CONTAINER = __DIR__ . '/' . self::CONTAINER_CLASS_NAME . '.php';
    /** @var ContainerInterface */
    private static $container;
    /**
     * This container uses the default values set in the whmcs integration settings page
     *
     * @return ContainerInterface
     * @throws Exception
     */
    public static function getDefaultContainer() : \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\DependencyInjection\ContainerInterface
    {
        if (\file_exists(self::CACHED_CONTAINER)) {
            require_once self::CACHED_CONTAINER;
            $className = self::CONTAINER_CLASS_NAME;
            self::$container = new $className();
        }
        if (self::$container === null) {
            $containerBuilder = self::initContainer();
            $containerBuilder->compile();
            self::$container = $containerBuilder;
            self::writeCache($containerBuilder);
        }
        return self::$container;
    }
    /**
     * This container uses the values based on the provided ClientWrapper
     *
     * @param SupportPalApiClientWrapper $supportPalApiClientWrapper
     * @return ContainerInterface
     * @throws Exception
     */
    public static function getCustomContainer(\SupportPal\WhmcsIntegration\SupportPalApiClientWrapper $supportPalApiClientWrapper) : \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\DependencyInjection\ContainerInterface
    {
        $containerBuilder = new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\DependencyInjection\ContainerBuilder();
        $containerBuilder->set(\SupportPal\WhmcsIntegration\SupportPalApiClientWrapper::class, $supportPalApiClientWrapper);
        $containerBuilder->merge(self::initContainer());
        $containerBuilder->compile();
        return $containerBuilder;
    }
    public static function clearCache() : void
    {
        if (!\file_exists(self::CACHED_CONTAINER)) {
            return;
        }
        \unlink(self::CACHED_CONTAINER);
    }
    /**
     * @return ContainerBuilder
     * @throws Exception
     */
    private static function initContainer() : \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\DependencyInjection\ContainerBuilder
    {
        $containerBuilder = new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\DependencyInjection\ContainerBuilder();
        $loader = new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\DependencyInjection\Loader\YamlFileLoader($containerBuilder, new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Config\FileLocator(__DIR__));
        $loader->load('../Resources/services.yml');
        return $containerBuilder;
    }
    private static function writeCache(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\DependencyInjection\ContainerBuilder $containerBuilder) : void
    {
        $dumper = new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\DependencyInjection\Dumper\PhpDumper($containerBuilder);
        \file_put_contents(self::CACHED_CONTAINER, $dumper->dump(['class' => self::CONTAINER_CLASS_NAME]));
    }
}
