<?php

declare (strict_types=1);
namespace SupportPal\WhmcsIntegration\Repository;

use Closure;
use InvalidArgumentException;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\Exception;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\HttpResponseException;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\InvalidDataException;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\RuntimeException;
use SupportPal\WhmcsIntegration\Exception\InvalidApiResponse;
use SupportPal\WhmcsIntegration\Log\ExceptionLogger;
use SupportPal\WhmcsIntegration\Log\RequestLogger;
use SupportPal\WhmcsIntegration\SupportPalApiClientWrapper;
/**
 * Class BaseRepository
 * @package SupportPal\WhmcsIntegration\Repository
 */
abstract class BaseRepository
{
    /** @var SupportPalApiClientWrapper */
    private $supportPalApiClientWrapper;
    /** @var RequestLogger */
    private $logger;
    /** @var ExceptionLogger */
    private $exceptionLogger;
    /**
     * BaseRepository constructor.
     * @param SupportPalApiClientWrapper $apiClientWrapper
     * @param RequestLogger $logger
     * @param ExceptionLogger $exceptionLogger
     */
    public function __construct(\SupportPal\WhmcsIntegration\SupportPalApiClientWrapper $apiClientWrapper, \SupportPal\WhmcsIntegration\Log\RequestLogger $logger, \SupportPal\WhmcsIntegration\Log\ExceptionLogger $exceptionLogger)
    {
        $this->supportPalApiClientWrapper = $apiClientWrapper;
        $this->logger = $logger;
        $this->exceptionLogger = $exceptionLogger;
    }
    /**
     * @param Closure $closure
     * @param string $endpoint
     * @return mixed
     * @throws InvalidApiResponse
     */
    protected function callAndMapException(\Closure $closure, string $endpoint)
    {
        try {
            return $closure();
        } catch (\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\HttpResponseException $httpResponseException) {
            $this->logger->log($httpResponseException, $this->getActionName($endpoint), $httpResponseException->getRequest(), $httpResponseException->getResponse());
            throw new \SupportPal\WhmcsIntegration\Exception\InvalidApiResponse($httpResponseException->getMessage(), $httpResponseException->getCode(), $httpResponseException);
        } catch (\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\Exception|\InvalidArgumentException|\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\RuntimeException $exception) {
            $this->exceptionLogger->log($exception, $this->getActionName($endpoint), $exception instanceof \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\InvalidDataException ? $exception->getData() : []);
            throw new \SupportPal\WhmcsIntegration\Exception\InvalidApiResponse($exception->getMessage(), $exception->getCode(), $exception);
        }
    }
    /**
     * @param string $endpoint
     * @return string
     */
    private function getActionName(string $endpoint) : string
    {
        return static::class . '::' . $endpoint;
    }
    /**
     * @return SupportPalApiClientWrapper
     */
    protected function getApiClientWrapper() : \SupportPal\WhmcsIntegration\SupportPalApiClientWrapper
    {
        return $this->supportPalApiClientWrapper;
    }
}
