<?php

namespace SupportPal\WhmcsIntegration\Vendor\Doctrine\Common\Cache;

use function array_combine;
use function array_key_exists;
use function array_map;
use function sprintf;
/**
 * Base class for cache provider implementations.
 */
abstract class CacheProvider implements \SupportPal\WhmcsIntegration\Vendor\Doctrine\Common\Cache\Cache, \SupportPal\WhmcsIntegration\Vendor\Doctrine\Common\Cache\FlushableCache, \SupportPal\WhmcsIntegration\Vendor\Doctrine\Common\Cache\ClearableCache, \SupportPal\WhmcsIntegration\Vendor\Doctrine\Common\Cache\MultiOperationCache
{
    public const DOCTRINE_NAMESPACE_CACHEKEY = 'DoctrineNamespaceCacheKey[%s]';
    /**
     * The namespace to prefix all cache ids with.
     *
     * @var string
     */
    private $namespace = '';
    /**
     * The namespace version.
     *
     * @var int|null
     */
    private $namespaceVersion;
    /**
     * Sets the namespace to prefix all cache ids with.
     *
     * @param string $namespace
     *
     * @return void
     */
    public function setNamespace($namespace)
    {
        $this->namespace = (string) $namespace;
        $this->namespaceVersion = null;
    }
    /**
     * Retrieves the namespace that prefixes all cache ids.
     *
     * @return string
     */
    public function getNamespace()
    {
        return $this->namespace;
    }
    /**
     * {@inheritdoc}
     */
    public function fetch($id)
    {
        return $this->doFetch($this->getNamespacedId($id));
    }
    /**
     * {@inheritdoc}
     */
    public function fetchMultiple(array $keys)
    {
        if (empty($keys)) {
            return [];
        }
        // note: the array_combine() is in place to keep an association between our $keys and the $namespacedKeys
        $namespacedKeys = \array_combine($keys, \array_map([$this, 'getNamespacedId'], $keys));
        $items = $this->doFetchMultiple($namespacedKeys);
        $foundItems = [];
        // no internal array function supports this sort of mapping: needs to be iterative
        // this filters and combines keys in one pass
        foreach ($namespacedKeys as $requestedKey => $namespacedKey) {
            if (!isset($items[$namespacedKey]) && !\array_key_exists($namespacedKey, $items)) {
                continue;
            }
            $foundItems[$requestedKey] = $items[$namespacedKey];
        }
        return $foundItems;
    }
    /**
     * {@inheritdoc}
     */
    public function saveMultiple(array $keysAndValues, $lifetime = 0)
    {
        $namespacedKeysAndValues = [];
        foreach ($keysAndValues as $key => $value) {
            $namespacedKeysAndValues[$this->getNamespacedId($key)] = $value;
        }
        return $this->doSaveMultiple($namespacedKeysAndValues, $lifetime);
    }
    /**
     * {@inheritdoc}
     */
    public function contains($id)
    {
        return $this->doContains($this->getNamespacedId($id));
    }
    /**
     * {@inheritdoc}
     */
    public function save($id, $data, $lifeTime = 0)
    {
        return $this->doSave($this->getNamespacedId($id), $data, $lifeTime);
    }
    /**
     * {@inheritdoc}
     */
    public function deleteMultiple(array $keys)
    {
        return $this->doDeleteMultiple(\array_map([$this, 'getNamespacedId'], $keys));
    }
    /**
     * {@inheritdoc}
     */
    public function delete($id)
    {
        return $this->doDelete($this->getNamespacedId($id));
    }
    /**
     * {@inheritdoc}
     */
    public function getStats()
    {
        return $this->doGetStats();
    }
    /**
     * {@inheritDoc}
     */
    public function flushAll()
    {
        return $this->doFlush();
    }
    /**
     * {@inheritDoc}
     */
    public function deleteAll()
    {
        $namespaceCacheKey = $this->getNamespaceCacheKey();
        $namespaceVersion = $this->getNamespaceVersion() + 1;
        if ($this->doSave($namespaceCacheKey, $namespaceVersion)) {
            $this->namespaceVersion = $namespaceVersion;
            return \true;
        }
        return \false;
    }
    /**
     * Prefixes the passed id with the configured namespace value.
     *
     * @param string $id The id to namespace.
     *
     * @return string The namespaced id.
     */
    private function getNamespacedId(string $id) : string
    {
        $namespaceVersion = $this->getNamespaceVersion();
        return \sprintf('%s[%s][%s]', $this->namespace, $id, $namespaceVersion);
    }
    /**
     * Returns the namespace cache key.
     */
    private function getNamespaceCacheKey() : string
    {
        return \sprintf(self::DOCTRINE_NAMESPACE_CACHEKEY, $this->namespace);
    }
    /**
     * Returns the namespace version.
     */
    private function getNamespaceVersion() : int
    {
        if ($this->namespaceVersion !== null) {
            return $this->namespaceVersion;
        }
        $namespaceCacheKey = $this->getNamespaceCacheKey();
        $this->namespaceVersion = (int) $this->doFetch($namespaceCacheKey) ?: 1;
        return $this->namespaceVersion;
    }
    /**
     * Default implementation of doFetchMultiple. Each driver that supports multi-get should owerwrite it.
     *
     * @param string[] $keys Array of keys to retrieve from cache
     *
     * @return mixed[] Array of values retrieved for the given keys.
     */
    protected function doFetchMultiple(array $keys)
    {
        $returnValues = [];
        foreach ($keys as $key) {
            $item = $this->doFetch($key);
            if ($item === \false && !$this->doContains($key)) {
                continue;
            }
            $returnValues[$key] = $item;
        }
        return $returnValues;
    }
    /**
     * Fetches an entry from the cache.
     *
     * @param string $id The id of the cache entry to fetch.
     *
     * @return mixed|false The cached data or FALSE, if no cache entry exists for the given id.
     */
    protected abstract function doFetch($id);
    /**
     * Tests if an entry exists in the cache.
     *
     * @param string $id The cache id of the entry to check for.
     *
     * @return bool TRUE if a cache entry exists for the given cache id, FALSE otherwise.
     */
    protected abstract function doContains($id);
    /**
     * Default implementation of doSaveMultiple. Each driver that supports multi-put should override it.
     *
     * @param mixed[] $keysAndValues Array of keys and values to save in cache
     * @param int     $lifetime      The lifetime. If != 0, sets a specific lifetime for these
     *                               cache entries (0 => infinite lifeTime).
     *
     * @return bool TRUE if the operation was successful, FALSE if it wasn't.
     */
    protected function doSaveMultiple(array $keysAndValues, $lifetime = 0)
    {
        $success = \true;
        foreach ($keysAndValues as $key => $value) {
            if ($this->doSave($key, $value, $lifetime)) {
                continue;
            }
            $success = \false;
        }
        return $success;
    }
    /**
     * Puts data into the cache.
     *
     * @param string $id       The cache id.
     * @param string $data     The cache entry/data.
     * @param int    $lifeTime The lifetime. If != 0, sets a specific lifetime for this
     *                           cache entry (0 => infinite lifeTime).
     *
     * @return bool TRUE if the entry was successfully stored in the cache, FALSE otherwise.
     */
    protected abstract function doSave($id, $data, $lifeTime = 0);
    /**
     * Default implementation of doDeleteMultiple. Each driver that supports multi-delete should override it.
     *
     * @param string[] $keys Array of keys to delete from cache
     *
     * @return bool TRUE if the operation was successful, FALSE if it wasn't
     */
    protected function doDeleteMultiple(array $keys)
    {
        $success = \true;
        foreach ($keys as $key) {
            if ($this->doDelete($key)) {
                continue;
            }
            $success = \false;
        }
        return $success;
    }
    /**
     * Deletes a cache entry.
     *
     * @param string $id The cache id.
     *
     * @return bool TRUE if the cache entry was successfully deleted, FALSE otherwise.
     */
    protected abstract function doDelete($id);
    /**
     * Flushes all cache entries.
     *
     * @return bool TRUE if the cache entries were successfully flushed, FALSE otherwise.
     */
    protected abstract function doFlush();
    /**
     * Retrieves cached information from the data store.
     *
     * @return mixed[]|null An associative array with server's statistics if available, NULL otherwise.
     */
    protected abstract function doGetStats();
}
