<?php

namespace SupportPal\WhmcsIntegration\Vendor\Kevinrob\GuzzleCache\Strategy;

use SupportPal\WhmcsIntegration\Vendor\Kevinrob\GuzzleCache\CacheEntry;
use SupportPal\WhmcsIntegration\Vendor\Kevinrob\GuzzleCache\KeyValueHttpHeader;
use SupportPal\WhmcsIntegration\Vendor\Kevinrob\GuzzleCache\Storage\CacheStorageInterface;
use SupportPal\WhmcsIntegration\Vendor\Psr\Http\Message\RequestInterface;
use SupportPal\WhmcsIntegration\Vendor\Psr\Http\Message\ResponseInterface;
/**
 * This strategy represents a "greedy" HTTP client.
 *
 * It can be used to cache responses in spite of any cache related response headers,
 * but it SHOULDN'T be used unless absolutely necessary, e.g. when accessing
 * badly designed APIs without Cache control.
 *
 * Obviously, this follows no RFC :(.
 */
class GreedyCacheStrategy extends \SupportPal\WhmcsIntegration\Vendor\Kevinrob\GuzzleCache\Strategy\PrivateCacheStrategy
{
    const HEADER_TTL = 'X-Kevinrob-GuzzleCache-TTL';
    /**
     * @var int
     */
    protected $defaultTtl;
    /**
     * @var KeyValueHttpHeader
     */
    private $varyHeaders;
    public function __construct(\SupportPal\WhmcsIntegration\Vendor\Kevinrob\GuzzleCache\Storage\CacheStorageInterface $cache = null, $defaultTtl, \SupportPal\WhmcsIntegration\Vendor\Kevinrob\GuzzleCache\KeyValueHttpHeader $varyHeaders = null)
    {
        $this->defaultTtl = $defaultTtl;
        $this->varyHeaders = $varyHeaders;
        parent::__construct($cache);
    }
    protected function getCacheKey(\SupportPal\WhmcsIntegration\Vendor\Psr\Http\Message\RequestInterface $request, \SupportPal\WhmcsIntegration\Vendor\Kevinrob\GuzzleCache\KeyValueHttpHeader $varyHeaders = null)
    {
        if (null === $varyHeaders || $varyHeaders->isEmpty()) {
            return \hash('sha256', 'greedy' . $request->getMethod() . $request->getUri());
        }
        $cacheHeaders = [];
        foreach ($varyHeaders as $key => $value) {
            if ($request->hasHeader($key)) {
                $cacheHeaders[$key] = $request->getHeader($key);
            }
        }
        return \hash('sha256', 'greedy' . $request->getMethod() . $request->getUri() . \json_encode($cacheHeaders));
    }
    public function cache(\SupportPal\WhmcsIntegration\Vendor\Psr\Http\Message\RequestInterface $request, \SupportPal\WhmcsIntegration\Vendor\Psr\Http\Message\ResponseInterface $response)
    {
        $warningMessage = \sprintf('%d - "%s" "%s"', 299, 'Cached although the response headers indicate not to do it!', (new \DateTime())->format(\DateTime::RFC1123));
        $response = $response->withAddedHeader('Warning', $warningMessage);
        if ($cacheObject = $this->getCacheObject($request, $response)) {
            return $this->storage->save($this->getCacheKey($request, $this->varyHeaders), $cacheObject);
        }
        return \false;
    }
    protected function getCacheObject(\SupportPal\WhmcsIntegration\Vendor\Psr\Http\Message\RequestInterface $request, \SupportPal\WhmcsIntegration\Vendor\Psr\Http\Message\ResponseInterface $response)
    {
        if (!\array_key_exists($response->getStatusCode(), $this->statusAccepted)) {
            // Don't cache it
            return null;
        }
        if (null !== $this->varyHeaders && $this->varyHeaders->has('*')) {
            // This will never match with a request
            return;
        }
        $response = $response->withoutHeader('Etag')->withoutHeader('Last-Modified');
        $ttl = $this->defaultTtl;
        if ($request->hasHeader(self::HEADER_TTL)) {
            $ttlHeaderValues = $request->getHeader(self::HEADER_TTL);
            $ttl = (int) \reset($ttlHeaderValues);
        }
        return new \SupportPal\WhmcsIntegration\Vendor\Kevinrob\GuzzleCache\CacheEntry($request->withoutHeader(self::HEADER_TTL), $response, new \DateTime(\sprintf('+%d seconds', $ttl)));
    }
    public function fetch(\SupportPal\WhmcsIntegration\Vendor\Psr\Http\Message\RequestInterface $request)
    {
        $cache = $this->storage->fetch($this->getCacheKey($request, $this->varyHeaders));
        return $cache;
    }
    /**
     * {@inheritdoc}
     */
    public function delete(\SupportPal\WhmcsIntegration\Vendor\Psr\Http\Message\RequestInterface $request)
    {
        return $this->storage->delete($this->getCacheKey($request));
    }
}
