<?php

declare (strict_types=1);
namespace SupportPal\WhmcsIntegration\Repository\SelfService;

use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\InvalidArgumentException;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Collection\Collection;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Shared\Settings;
use SupportPal\WhmcsIntegration\Exception\InvalidApiResponse;
use SupportPal\WhmcsIntegration\Helper\DateTimeFormatter;
use SupportPal\WhmcsIntegration\Log\ExceptionLogger;
use SupportPal\WhmcsIntegration\Log\RequestLogger;
use SupportPal\WhmcsIntegration\Service\Converter\ModelToArrayConverter;
use SupportPal\WhmcsIntegration\SupportPalApiClientWrapper;
use function array_merge;
class CommentRepository extends \SupportPal\WhmcsIntegration\Repository\SelfService\BaseSelfServiceRepository
{
    /** @var DateTimeFormatter */
    private $dateTimeFormatter;
    /** @var ModelToArrayConverter */
    private $modelToArrayConverter;
    /**
     * CommentRepository constructor.
     * @param SupportPalApiClientWrapper $apiClientWrapper
     * @param DateTimeFormatter $dateTimeFormatter
     * @param ModelToArrayConverter $modelToArrayConverter
     * @param RequestLogger $logger
     * @param ExceptionLogger $exceptionLogger
     */
    public function __construct(\SupportPal\WhmcsIntegration\SupportPalApiClientWrapper $apiClientWrapper, \SupportPal\WhmcsIntegration\Helper\DateTimeFormatter $dateTimeFormatter, \SupportPal\WhmcsIntegration\Service\Converter\ModelToArrayConverter $modelToArrayConverter, \SupportPal\WhmcsIntegration\Log\RequestLogger $logger, \SupportPal\WhmcsIntegration\Log\ExceptionLogger $exceptionLogger)
    {
        parent::__construct($apiClientWrapper, $logger, $exceptionLogger);
        $this->dateTimeFormatter = $dateTimeFormatter;
        $this->modelToArrayConverter = $modelToArrayConverter;
    }
    /**
     * @param mixed[] $data
     * @return Collection
     * @throws InvalidApiResponse
     * @throws InvalidArgumentException
     */
    public function findBy(array $data) : \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Collection\Collection
    {
        return $this->callAndMapException(function () use($data) {
            return $this->getApi()->getComments($data);
        }, 'getComments');
    }
    /**
     * @param mixed[] $data
     * @param Settings $settings
     * @return mixed[]
     * @throws InvalidApiResponse
     * @throws InvalidArgumentException
     */
    public function getCommentTree(array $data, \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Shared\Settings $settings) : array
    {
        $allComments = $this->modelToArrayConverter->convertMany($this->fetchAllComments($data)->getModels());
        return !empty($allComments) ? $this->buildCommentTree($allComments, $settings) : [];
    }
    /**
     * @param mixed[] $data
     * @return Collection
     * @throws InvalidApiResponse
     * @throws InvalidArgumentException
     */
    private function fetchAllComments(array $data) : \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Collection\Collection
    {
        $data['start'] = $data['start'] ?? 1;
        $collection = new \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Collection\Collection();
        while (\true) {
            $commentsCollection = $this->findBy($data);
            if ($commentsCollection->isEmpty()) {
                break;
            }
            $data['start'] += $commentsCollection->getModelsCount();
            $collection = $collection->merge($commentsCollection);
        }
        return $collection;
    }
    /**
     * Builds the tree for comments recursively
     *
     * @param mixed[] $comments
     * @param Settings $settings
     * @param int|null $commentId
     * @return mixed[]
     */
    private function buildCommentTree(array $comments, \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Model\Shared\Settings $settings, ?int $commentId = null) : array
    {
        $finalComments = [];
        $dateFormat = (string) $settings->get('date_format', 'd/m/Y');
        $timeFormat = (string) $settings->get('time_format', 'g:i A');
        foreach ($comments as $key => $value) {
            if ($value['root_parent_id'] !== $commentId) {
                continue;
            }
            $comments[$key]['created_at'] = $this->dateTimeFormatter->format($dateFormat, $timeFormat, $value['created_at']);
            /** Add to the returning array */
            $finalComments[] = $comments[$key];
            /** Find any children and add them to the array at this point */
            $finalComments = \array_merge($finalComments, $this->buildCommentTree($comments, $settings, $value['id']));
        }
        return $finalComments;
    }
}
