<?php

declare (strict_types=1);
namespace SupportPal\WhmcsIntegration;

use Exception;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Config\ApiContext;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Config\RequestDefaults;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\InvalidArgumentException;
use SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\SupportPal;
use SupportPal\WhmcsIntegration\Factory\RequestDefaultsFactory;
use SupportPal\WhmcsIntegration\Factory\SupportPalFactory;
use SupportPal\WhmcsIntegration\Helper\TransientDataHelper;
use SupportPal\WhmcsIntegration\Repository\Whmcs\ModSupportPalRepository;
use WHMCS\Application\Support\Facades\Lang;
use function array_merge;
use function is_array;
use function json_decode;
/**
 * Class ApiClient
 * @package SupportPal\WhmcsIntegration
 */
class SupportPalApiClientWrapper
{
    /** @var SupportPal */
    private $supportPal;
    /** @var ModSupportPalRepository */
    private $modSupportPalRepository;
    /** @var RequestDefaultsFactory */
    private $requestDefaultsFactory;
    /** @var TransientDataHelper */
    private $transientDataHelper;
    /** @var SupportPalFactory */
    private $supportpalFactory;
    /**
     * ApiClient constructor.
     * @param SupportPalFactory $supportpalFactory
     * @param RequestDefaultsFactory $requestDefaultsFactory
     * @param ModSupportPalRepository $modSupportPalRepository
     * @param TransientDataHelper $transientDataHelper
     * @param SupportPal|null $supportPal
     * @throws Exception
     */
    public function __construct(\SupportPal\WhmcsIntegration\Factory\SupportPalFactory $supportpalFactory, \SupportPal\WhmcsIntegration\Factory\RequestDefaultsFactory $requestDefaultsFactory, \SupportPal\WhmcsIntegration\Repository\Whmcs\ModSupportPalRepository $modSupportPalRepository, \SupportPal\WhmcsIntegration\Helper\TransientDataHelper $transientDataHelper, ?\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\SupportPal $supportPal = null)
    {
        $this->supportpalFactory = $supportpalFactory;
        $this->requestDefaultsFactory = $requestDefaultsFactory;
        $this->modSupportPalRepository = $modSupportPalRepository;
        $this->transientDataHelper = $transientDataHelper;
        $this->supportPal = $supportPal ?? $this->getSupportPalInstance();
    }
    /**
     * @return SupportPal
     * @throws Exception
     */
    private function getSupportPalInstance() : \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\SupportPal
    {
        $baseUrl = $this->modSupportPalRepository->getDefaultBaseUrl();
        $baseUrl = $baseUrl === null ? '' : $baseUrl;
        $apiToken = $this->modSupportPalRepository->getDefaultApiToken();
        $apiToken = $apiToken === null ? '' : $apiToken;
        try {
            $apiContext = \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Config\ApiContext::createFromUrl($baseUrl, $apiToken);
        } catch (\SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Exception\InvalidArgumentException $_) {
            /**
             * @todo should be replaced with exception handling when we encapsulate ClientArea
             */
            $apiContext = new \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Config\ApiContext('', '');
        }
        $requestDefaults = $this->requestDefaultsFactory->create(['whmcs' => \true], []);
        // Set the language.
        if ($language = \WHMCS\Application\Support\Facades\Lang::getName()) {
            foreach ($this->getLocales() as $locale) {
                if ($locale['language'] !== $language || !isset($locale['languageCode'])) {
                    continue;
                }
                $this->appendDefaultValue('lang', $locale['languageCode'], $requestDefaults);
                break;
            }
        }
        // Set the brand ID.
        $defaultBrand = $this->modSupportPalRepository->getDefaultBrand();
        if (!empty($defaultBrand)) {
            $this->appendDefaultValue('brand_id', $defaultBrand, $requestDefaults);
            return $this->supportpalFactory->create($apiContext, $requestDefaults);
        }
        $supportPal = $this->supportpalFactory->create($apiContext, $requestDefaults);
        $brandId = $supportPal->getCoreApi()->getSettings()->get('default_brand', 1);
        $this->appendDefaultValue('brand_id', $brandId, $requestDefaults);
        /**
         * SupportPal class is immutable, thus any change to parameters requires a new instance
         */
        return $this->supportpalFactory->create($apiContext, $requestDefaults);
    }
    /**
     * @return SupportPal
     */
    public function getSupportPal() : \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\SupportPal
    {
        return $this->supportPal;
    }
    /**
     * @param string $key
     * @param mixed $value
     * @param RequestDefaults $requestDefaults
     */
    private function appendDefaultValue(string $key, $value, \SupportPal\WhmcsIntegration\Vendor\SupportPal\ApiClient\Config\RequestDefaults $requestDefaults) : void
    {
        $requestDefaults->setDefaultParameters(\array_merge($requestDefaults->getDefaultParameters(), [$key => $value]));
    }
    /**
     * @return array<array<string,string>>
     */
    private function getLocales() : array
    {
        $data = $this->transientDataHelper->retrieve('WHMCS\\Language\\ClientLanguageLocales');
        if ($data === null) {
            return [];
        }
        $locales = \json_decode($data, \true);
        if (!\is_array($locales)) {
            return [];
        }
        return $locales['locales'] ?? [];
    }
}
