<?php

declare (strict_types=1);
/**
 * This file is part of phpDocumentor.
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 *
 * @link      http://phpdoc.org
 */
namespace SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection;

use ArrayIterator;
use InvalidArgumentException;
use SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\PseudoTypes\IntegerRange;
use SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\PseudoTypes\List_;
use SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Array_;
use SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\ArrayKey;
use SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\ClassString;
use SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Collection;
use SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Compound;
use SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Context;
use SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Expression;
use SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Integer;
use SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\InterfaceString;
use SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Intersection;
use SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Iterable_;
use SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Nullable;
use SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Object_;
use SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\String_;
use RuntimeException;
use function array_key_exists;
use function array_pop;
use function array_values;
use function class_exists;
use function class_implements;
use function count;
use function current;
use function end;
use function in_array;
use function is_numeric;
use function key;
use function preg_split;
use function strpos;
use function strtolower;
use function trim;
use const PREG_SPLIT_DELIM_CAPTURE;
use const PREG_SPLIT_NO_EMPTY;
final class TypeResolver
{
    /** @var string Definition of the ARRAY operator for types */
    private const OPERATOR_ARRAY = '[]';
    /** @var string Definition of the NAMESPACE operator in PHP */
    private const OPERATOR_NAMESPACE = '\\';
    /** @var int the iterator parser is inside a compound context */
    private const PARSER_IN_COMPOUND = 0;
    /** @var int the iterator parser is inside a nullable expression context */
    private const PARSER_IN_NULLABLE = 1;
    /** @var int the iterator parser is inside an array expression context */
    private const PARSER_IN_ARRAY_EXPRESSION = 2;
    /** @var int the iterator parser is inside a collection expression context */
    private const PARSER_IN_COLLECTION_EXPRESSION = 3;
    /**
     * @var array<string, string> List of recognized keywords and unto which Value Object they map
     * @psalm-var array<string, class-string<Type>>
     */
    private $keywords = ['string' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\String_::class, 'class-string' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\ClassString::class, 'interface-string' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\InterfaceString::class, 'html-escaped-string' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\PseudoTypes\HtmlEscapedString::class, 'lowercase-string' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\PseudoTypes\LowercaseString::class, 'non-empty-lowercase-string' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\PseudoTypes\NonEmptyLowercaseString::class, 'non-empty-string' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\PseudoTypes\NonEmptyString::class, 'numeric-string' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\PseudoTypes\NumericString::class, 'numeric' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\PseudoTypes\Numeric_::class, 'trait-string' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\PseudoTypes\TraitString::class, 'int' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Integer::class, 'integer' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Integer::class, 'positive-int' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\PseudoTypes\PositiveInteger::class, 'negative-int' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\PseudoTypes\NegativeInteger::class, 'bool' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Boolean::class, 'boolean' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Boolean::class, 'real' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Float_::class, 'float' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Float_::class, 'double' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Float_::class, 'object' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Object_::class, 'mixed' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Mixed_::class, 'array' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Array_::class, 'array-key' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\ArrayKey::class, 'resource' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Resource_::class, 'void' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Void_::class, 'null' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Null_::class, 'scalar' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Scalar::class, 'callback' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Callable_::class, 'callable' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Callable_::class, 'callable-string' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\PseudoTypes\CallableString::class, 'false' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\PseudoTypes\False_::class, 'true' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\PseudoTypes\True_::class, 'literal-string' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\PseudoTypes\LiteralString::class, 'self' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Self_::class, '$this' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\This::class, 'static' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Static_::class, 'parent' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Parent_::class, 'iterable' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Iterable_::class, 'never' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Never_::class, 'list' => \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\PseudoTypes\List_::class];
    /**
     * @var FqsenResolver
     * @psalm-readonly
     */
    private $fqsenResolver;
    /**
     * Initializes this TypeResolver with the means to create and resolve Fqsen objects.
     */
    public function __construct(?\SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\FqsenResolver $fqsenResolver = null)
    {
        $this->fqsenResolver = $fqsenResolver ?: new \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\FqsenResolver();
    }
    /**
     * Analyzes the given type and returns the FQCN variant.
     *
     * When a type is provided this method checks whether it is not a keyword or
     * Fully Qualified Class Name. If so it will use the given namespace and
     * aliases to expand the type to a FQCN representation.
     *
     * This method only works as expected if the namespace and aliases are set;
     * no dynamic reflection is being performed here.
     *
     * @uses Context::getNamespaceAliases() to check whether the first part of the relative type name should not be
     * replaced with another namespace.
     * @uses Context::getNamespace()        to determine with what to prefix the type name.
     *
     * @param string $type The relative or absolute type.
     */
    public function resolve(string $type, ?\SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Context $context = null) : \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Type
    {
        $type = \trim($type);
        if (!$type) {
            throw new \InvalidArgumentException('Attempted to resolve "' . $type . '" but it appears to be empty');
        }
        if ($context === null) {
            $context = new \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Context('');
        }
        // split the type string into tokens `|`, `?`, `<`, `>`, `,`, `(`, `)`, `[]`, '<', '>' and type names
        $tokens = \preg_split('/(\\||\\?|<|>|&|, ?|\\(|\\)|\\[\\]+)/', $type, -1, \PREG_SPLIT_NO_EMPTY | \PREG_SPLIT_DELIM_CAPTURE);
        if ($tokens === \false) {
            throw new \InvalidArgumentException('Unable to split the type string "' . $type . '" into tokens');
        }
        /** @var ArrayIterator<int, string|null> $tokenIterator */
        $tokenIterator = new \ArrayIterator($tokens);
        return $this->parseTypes($tokenIterator, $context, self::PARSER_IN_COMPOUND);
    }
    /**
     * Analyse each tokens and creates types
     *
     * @param ArrayIterator<int, string|null> $tokens        the iterator on tokens
     * @param int                        $parserContext on of self::PARSER_* constants, indicating
     * the context where we are in the parsing
     */
    private function parseTypes(\ArrayIterator $tokens, \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Context $context, int $parserContext) : \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Type
    {
        $types = [];
        $token = '';
        $compoundToken = '|';
        while ($tokens->valid()) {
            $token = $tokens->current();
            if ($token === null) {
                throw new \RuntimeException('Unexpected nullable character');
            }
            if ($token === '|' || $token === '&') {
                if (\count($types) === 0) {
                    throw new \RuntimeException('A type is missing before a type separator');
                }
                if (!\in_array($parserContext, [self::PARSER_IN_COMPOUND, self::PARSER_IN_ARRAY_EXPRESSION, self::PARSER_IN_COLLECTION_EXPRESSION], \true)) {
                    throw new \RuntimeException('Unexpected type separator');
                }
                $compoundToken = $token;
                $tokens->next();
            } elseif ($token === '?') {
                if (!\in_array($parserContext, [self::PARSER_IN_COMPOUND, self::PARSER_IN_ARRAY_EXPRESSION, self::PARSER_IN_COLLECTION_EXPRESSION], \true)) {
                    throw new \RuntimeException('Unexpected nullable character');
                }
                $tokens->next();
                $type = $this->parseTypes($tokens, $context, self::PARSER_IN_NULLABLE);
                $types[] = new \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Nullable($type);
            } elseif ($token === '(') {
                $tokens->next();
                $type = $this->parseTypes($tokens, $context, self::PARSER_IN_ARRAY_EXPRESSION);
                $token = $tokens->current();
                if ($token === null) {
                    // Someone did not properly close their array expression ..
                    break;
                }
                $tokens->next();
                $resolvedType = new \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Expression($type);
                $types[] = $resolvedType;
            } elseif ($parserContext === self::PARSER_IN_ARRAY_EXPRESSION && isset($token[0]) && $token[0] === ')') {
                break;
            } elseif ($token === '<') {
                if (\count($types) === 0) {
                    throw new \RuntimeException('Unexpected collection operator "<", class name is missing');
                }
                $classType = \array_pop($types);
                if ($classType !== null) {
                    if ((string) $classType === 'class-string') {
                        $types[] = $this->resolveClassString($tokens, $context);
                    } elseif ((string) $classType === 'int') {
                        $types[] = $this->resolveIntRange($tokens);
                    } elseif ((string) $classType === 'interface-string') {
                        $types[] = $this->resolveInterfaceString($tokens, $context);
                    } else {
                        $types[] = $this->resolveCollection($tokens, $classType, $context);
                    }
                }
                $tokens->next();
            } elseif ($parserContext === self::PARSER_IN_COLLECTION_EXPRESSION && ($token === '>' || \trim($token) === ',')) {
                break;
            } elseif ($token === self::OPERATOR_ARRAY) {
                \end($types);
                $last = \key($types);
                if ($last === null) {
                    throw new \InvalidArgumentException('Unexpected array operator');
                }
                $lastItem = $types[$last];
                if ($lastItem instanceof \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Expression) {
                    $lastItem = $lastItem->getValueType();
                }
                $types[$last] = new \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Array_($lastItem);
                $tokens->next();
            } else {
                $type = $this->resolveSingleType($token, $context);
                $tokens->next();
                if ($parserContext === self::PARSER_IN_NULLABLE) {
                    return $type;
                }
                $types[] = $type;
            }
        }
        if ($token === '|' || $token === '&') {
            throw new \RuntimeException('A type is missing after a type separator');
        }
        if (\count($types) === 0) {
            if ($parserContext === self::PARSER_IN_NULLABLE) {
                throw new \RuntimeException('A type is missing after a nullable character');
            }
            if ($parserContext === self::PARSER_IN_ARRAY_EXPRESSION) {
                throw new \RuntimeException('A type is missing in an array expression');
            }
            if ($parserContext === self::PARSER_IN_COLLECTION_EXPRESSION) {
                throw new \RuntimeException('A type is missing in a collection expression');
            }
        } elseif (\count($types) === 1) {
            return \current($types);
        }
        if ($compoundToken === '|') {
            return new \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Compound(\array_values($types));
        }
        return new \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Intersection(\array_values($types));
    }
    /**
     * resolve the given type into a type object
     *
     * @param string $type the type string, representing a single type
     *
     * @return Type|Array_|Object_
     *
     * @psalm-mutation-free
     */
    private function resolveSingleType(string $type, \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Context $context) : object
    {
        switch (\true) {
            case $this->isKeyword($type):
                return $this->resolveKeyword($type);
            case $this->isFqsen($type):
                return $this->resolveTypedObject($type);
            case $this->isPartialStructuralElementName($type):
                return $this->resolveTypedObject($type, $context);
            // @codeCoverageIgnoreStart
            default:
                // I haven't got the foggiest how the logic would come here but added this as a defense.
                throw new \RuntimeException('Unable to resolve type "' . $type . '", there is no known method to resolve it');
        }
        // @codeCoverageIgnoreEnd
    }
    /**
     * Adds a keyword to the list of Keywords and associates it with a specific Value Object.
     *
     * @psalm-param class-string<Type> $typeClassName
     */
    public function addKeyword(string $keyword, string $typeClassName) : void
    {
        if (!\class_exists($typeClassName)) {
            throw new \InvalidArgumentException('The Value Object that needs to be created with a keyword "' . $keyword . '" must be an existing class' . ' but we could not find the class ' . $typeClassName);
        }
        $interfaces = \class_implements($typeClassName);
        if ($interfaces === \false) {
            throw new \InvalidArgumentException('The Value Object that needs to be created with a keyword "' . $keyword . '" must be an existing class' . ' but we could not find the class ' . $typeClassName);
        }
        if (!\in_array(\SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Type::class, $interfaces, \true)) {
            throw new \InvalidArgumentException('The class "' . $typeClassName . '" must implement the interface "phpDocumentor\\Reflection\\Type"');
        }
        $this->keywords[$keyword] = $typeClassName;
    }
    /**
     * Detects whether the given type represents a PHPDoc keyword.
     *
     * @param string $type A relative or absolute type as defined in the phpDocumentor documentation.
     *
     * @psalm-mutation-free
     */
    private function isKeyword(string $type) : bool
    {
        return \array_key_exists(\strtolower($type), $this->keywords);
    }
    /**
     * Detects whether the given type represents a relative structural element name.
     *
     * @param string $type A relative or absolute type as defined in the phpDocumentor documentation.
     *
     * @psalm-mutation-free
     */
    private function isPartialStructuralElementName(string $type) : bool
    {
        return isset($type[0]) && $type[0] !== self::OPERATOR_NAMESPACE && !$this->isKeyword($type);
    }
    /**
     * Tests whether the given type is a Fully Qualified Structural Element Name.
     *
     * @psalm-mutation-free
     */
    private function isFqsen(string $type) : bool
    {
        return \strpos($type, self::OPERATOR_NAMESPACE) === 0;
    }
    /**
     * Resolves the given keyword (such as `string`) into a Type object representing that keyword.
     *
     * @psalm-mutation-free
     */
    private function resolveKeyword(string $type) : \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Type
    {
        $className = $this->keywords[\strtolower($type)];
        return new $className();
    }
    /**
     * Resolves the given FQSEN string into an FQSEN object.
     *
     * @psalm-mutation-free
     */
    private function resolveTypedObject(string $type, ?\SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Context $context = null) : \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Object_
    {
        return new \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Object_($this->fqsenResolver->resolve($type, $context));
    }
    /**
     * Resolves class string
     *
     * @param ArrayIterator<int, (string|null)> $tokens
     */
    private function resolveClassString(\ArrayIterator $tokens, \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Context $context) : \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Type
    {
        $tokens->next();
        $classType = $this->parseTypes($tokens, $context, self::PARSER_IN_COLLECTION_EXPRESSION);
        if (!$classType instanceof \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Object_ || $classType->getFqsen() === null) {
            throw new \RuntimeException($classType . ' is not a class string');
        }
        $token = $tokens->current();
        if ($token !== '>') {
            if (empty($token)) {
                throw new \RuntimeException('class-string: ">" is missing');
            }
            throw new \RuntimeException('Unexpected character "' . $token . '", ">" is missing');
        }
        return new \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\ClassString($classType->getFqsen());
    }
    /**
     * Resolves integer ranges
     *
     * @param ArrayIterator<int, (string|null)> $tokens
     */
    private function resolveIntRange(\ArrayIterator $tokens) : \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Type
    {
        $tokens->next();
        $token = '';
        $minValue = null;
        $maxValue = null;
        $commaFound = \false;
        $tokenCounter = 0;
        while ($tokens->valid()) {
            $tokenCounter++;
            $token = $tokens->current();
            if ($token === null) {
                throw new \RuntimeException('Unexpected nullable character');
            }
            $token = \trim($token);
            if ($token === '>') {
                break;
            }
            if ($token === ',') {
                $commaFound = \true;
            }
            if ($commaFound === \false && $minValue === null) {
                if (\is_numeric($token) || $token === 'max' || $token === 'min') {
                    $minValue = $token;
                }
            }
            if ($commaFound === \true && $maxValue === null) {
                if (\is_numeric($token) || $token === 'max' || $token === 'min') {
                    $maxValue = $token;
                }
            }
            $tokens->next();
        }
        if ($token !== '>') {
            if (empty($token)) {
                throw new \RuntimeException('interface-string: ">" is missing');
            }
            throw new \RuntimeException('Unexpected character "' . $token . '", ">" is missing');
        }
        if ($minValue === null || $maxValue === null || $tokenCounter > 4) {
            throw new \RuntimeException('int<min,max> has not the correct format');
        }
        return new \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\PseudoTypes\IntegerRange($minValue, $maxValue);
    }
    /**
     * Resolves class string
     *
     * @param ArrayIterator<int, (string|null)> $tokens
     */
    private function resolveInterfaceString(\ArrayIterator $tokens, \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Context $context) : \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Type
    {
        $tokens->next();
        $classType = $this->parseTypes($tokens, $context, self::PARSER_IN_COLLECTION_EXPRESSION);
        if (!$classType instanceof \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Object_ || $classType->getFqsen() === null) {
            throw new \RuntimeException($classType . ' is not a interface string');
        }
        $token = $tokens->current();
        if ($token !== '>') {
            if (empty($token)) {
                throw new \RuntimeException('interface-string: ">" is missing');
            }
            throw new \RuntimeException('Unexpected character "' . $token . '", ">" is missing');
        }
        return new \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\InterfaceString($classType->getFqsen());
    }
    /**
     * Resolves the collection values and keys
     *
     * @param ArrayIterator<int, (string|null)> $tokens
     *
     * @return Array_|Iterable_|Collection
     */
    private function resolveCollection(\ArrayIterator $tokens, \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Type $classType, \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Context $context) : \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Type
    {
        $isArray = (string) $classType === 'array';
        $isIterable = (string) $classType === 'iterable';
        $isList = (string) $classType === 'list';
        // allow only "array", "iterable" or class name before "<"
        if (!$isArray && !$isIterable && !$isList && (!$classType instanceof \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Object_ || $classType->getFqsen() === null)) {
            throw new \RuntimeException($classType . ' is not a collection');
        }
        $tokens->next();
        $valueType = $this->parseTypes($tokens, $context, self::PARSER_IN_COLLECTION_EXPRESSION);
        $keyType = null;
        $token = $tokens->current();
        if ($token !== null && \trim($token) === ',' && !$isList) {
            // if we have a comma, then we just parsed the key type, not the value type
            $keyType = $valueType;
            if ($isArray) {
                // check the key type for an "array" collection. We allow only
                // strings or integers.
                if (!$keyType instanceof \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\ArrayKey && !$keyType instanceof \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\String_ && !$keyType instanceof \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Integer && !$keyType instanceof \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Compound) {
                    throw new \RuntimeException('An array can have only integers or strings as keys');
                }
                if ($keyType instanceof \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Compound) {
                    foreach ($keyType->getIterator() as $item) {
                        if (!$item instanceof \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\ArrayKey && !$item instanceof \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\String_ && !$item instanceof \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Integer) {
                            throw new \RuntimeException('An array can have only integers or strings as keys');
                        }
                    }
                }
            }
            $tokens->next();
            // now let's parse the value type
            $valueType = $this->parseTypes($tokens, $context, self::PARSER_IN_COLLECTION_EXPRESSION);
        }
        $token = $tokens->current();
        if ($token !== '>') {
            if (empty($token)) {
                throw new \RuntimeException('Collection: ">" is missing');
            }
            throw new \RuntimeException('Unexpected character "' . $token . '", ">" is missing');
        }
        if ($isArray) {
            return new \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Array_($valueType, $keyType);
        }
        if ($isIterable) {
            return new \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Iterable_($valueType, $keyType);
        }
        if ($isList) {
            return new \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\PseudoTypes\List_($valueType);
        }
        if ($classType instanceof \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Object_) {
            return $this->makeCollectionFromObject($classType, $valueType, $keyType);
        }
        throw new \RuntimeException('Invalid $classType provided');
    }
    /**
     * @psalm-pure
     */
    private function makeCollectionFromObject(\SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Object_ $object, \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Type $valueType, ?\SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Type $keyType = null) : \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Collection
    {
        return new \SupportPal\WhmcsIntegration\Vendor\phpDocumentor\Reflection\Types\Collection($object->getFqsen(), $valueType, $keyType);
    }
}
