<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\RateLimiter;

use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Request;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\RateLimiter\LimiterInterface;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\RateLimiter\Policy\NoLimiter;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\RateLimiter\RateLimit;
/**
 * An implementation of RequestRateLimiterInterface that
 * fits most use-cases.
 *
 * @author Wouter de Jong <wouter@wouterj.nl>
 */
abstract class AbstractRequestRateLimiter implements \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\RateLimiter\RequestRateLimiterInterface
{
    public function consume(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Request $request) : \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\RateLimiter\RateLimit
    {
        $limiters = $this->getLimiters($request);
        if (0 === \count($limiters)) {
            $limiters = [new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\RateLimiter\Policy\NoLimiter()];
        }
        $minimalRateLimit = null;
        foreach ($limiters as $limiter) {
            $rateLimit = $limiter->consume(1);
            $minimalRateLimit = $minimalRateLimit ? self::getMinimalRateLimit($minimalRateLimit, $rateLimit) : $rateLimit;
        }
        return $minimalRateLimit;
    }
    public function reset(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Request $request) : void
    {
        foreach ($this->getLimiters($request) as $limiter) {
            $limiter->reset();
        }
    }
    /**
     * @return LimiterInterface[] a set of limiters using keys extracted from the request
     */
    protected abstract function getLimiters(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Request $request) : array;
    private static function getMinimalRateLimit(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\RateLimiter\RateLimit $first, \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\RateLimiter\RateLimit $second) : \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\RateLimiter\RateLimit
    {
        if ($first->isAccepted() !== $second->isAccepted()) {
            return $first->isAccepted() ? $second : $first;
        }
        $firstRemainingTokens = $first->getRemainingTokens();
        $secondRemainingTokens = $second->getRemainingTokens();
        if ($firstRemainingTokens === $secondRemainingTokens) {
            return $first->getRetryAfter() < $second->getRetryAfter() ? $second : $first;
        }
        return $firstRemainingTokens > $secondRemainingTokens ? $second : $first;
    }
}
