<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Session\Storage\Handler;

use SupportPal\WhmcsIntegration\Vendor\Doctrine\DBAL\Configuration;
use SupportPal\WhmcsIntegration\Vendor\Doctrine\DBAL\DriverManager;
use SupportPal\WhmcsIntegration\Vendor\Doctrine\DBAL\Schema\DefaultSchemaManagerFactory;
use SupportPal\WhmcsIntegration\Vendor\Doctrine\DBAL\Tools\DsnParser;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Cache\Adapter\AbstractAdapter;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Cache\Traits\RedisClusterProxy;
use SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Cache\Traits\RedisProxy;
/**
 * @author Nicolas Grekas <p@tchwork.com>
 */
class SessionHandlerFactory
{
    /**
     * @param \Redis|\RedisArray|\RedisCluster|\Predis\ClientInterface|RedisProxy|RedisClusterProxy|\Memcached|\PDO|string $connection Connection or DSN
     */
    public static function createHandler($connection) : \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Session\Storage\Handler\AbstractSessionHandler
    {
        if (!\is_string($connection) && !\is_object($connection)) {
            throw new \TypeError(\sprintf('Argument 1 passed to "%s()" must be a string or a connection object, "%s" given.', __METHOD__, \get_debug_type($connection)));
        }
        if ($options = \is_string($connection) ? \parse_url($connection) : \false) {
            \parse_str($options['query'] ?? '', $options);
        }
        switch (\true) {
            case $connection instanceof \Redis:
            case $connection instanceof \RedisArray:
            case $connection instanceof \RedisCluster:
            case $connection instanceof \SupportPal\WhmcsIntegration\Vendor\Predis\ClientInterface:
            case $connection instanceof \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Cache\Traits\RedisProxy:
            case $connection instanceof \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Cache\Traits\RedisClusterProxy:
                return new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Session\Storage\Handler\RedisSessionHandler($connection);
            case $connection instanceof \Memcached:
                return new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Session\Storage\Handler\MemcachedSessionHandler($connection);
            case $connection instanceof \PDO:
                return new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Session\Storage\Handler\PdoSessionHandler($connection);
            case !\is_string($connection):
                throw new \InvalidArgumentException(\sprintf('Unsupported Connection: "%s".', \get_debug_type($connection)));
            case \str_starts_with($connection, 'file://'):
                $savePath = \substr($connection, 7);
                return new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Session\Storage\Handler\StrictSessionHandler(new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Session\Storage\Handler\NativeFileSessionHandler('' === $savePath ? null : $savePath));
            case \str_starts_with($connection, 'redis:'):
            case \str_starts_with($connection, 'rediss:'):
            case \str_starts_with($connection, 'memcached:'):
                if (!\class_exists(\SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Cache\Adapter\AbstractAdapter::class)) {
                    throw new \InvalidArgumentException('Unsupported Redis or Memcached DSN. Try running "composer require symfony/cache".');
                }
                $handlerClass = \str_starts_with($connection, 'memcached:') ? \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Session\Storage\Handler\MemcachedSessionHandler::class : \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Session\Storage\Handler\RedisSessionHandler::class;
                $connection = \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\Cache\Adapter\AbstractAdapter::createConnection($connection, ['lazy' => \true]);
                return new $handlerClass($connection, \array_intersect_key($options ?: [], ['prefix' => 1, 'ttl' => 1]));
            case \str_starts_with($connection, 'pdo_oci://'):
                if (!\class_exists(\SupportPal\WhmcsIntegration\Vendor\Doctrine\DBAL\DriverManager::class)) {
                    throw new \InvalidArgumentException('Unsupported PDO OCI DSN. Try running "composer require doctrine/dbal".');
                }
                $connection[3] = '-';
                $params = \class_exists(\SupportPal\WhmcsIntegration\Vendor\Doctrine\DBAL\Tools\DsnParser::class) ? (new \SupportPal\WhmcsIntegration\Vendor\Doctrine\DBAL\Tools\DsnParser())->parse($connection) : ['url' => $connection];
                $config = new \SupportPal\WhmcsIntegration\Vendor\Doctrine\DBAL\Configuration();
                if (\class_exists(\SupportPal\WhmcsIntegration\Vendor\Doctrine\DBAL\Schema\DefaultSchemaManagerFactory::class)) {
                    $config->setSchemaManagerFactory(new \SupportPal\WhmcsIntegration\Vendor\Doctrine\DBAL\Schema\DefaultSchemaManagerFactory());
                }
                $connection = \SupportPal\WhmcsIntegration\Vendor\Doctrine\DBAL\DriverManager::getConnection($params, $config);
                // The condition should be removed once support for DBAL <3.3 is dropped
                $connection = \method_exists($connection, 'getNativeConnection') ? $connection->getNativeConnection() : $connection->getWrappedConnection();
            // no break;
            case \str_starts_with($connection, 'mssql://'):
            case \str_starts_with($connection, 'mysql://'):
            case \str_starts_with($connection, 'mysql2://'):
            case \str_starts_with($connection, 'pgsql://'):
            case \str_starts_with($connection, 'postgres://'):
            case \str_starts_with($connection, 'postgresql://'):
            case \str_starts_with($connection, 'sqlsrv://'):
            case \str_starts_with($connection, 'sqlite://'):
            case \str_starts_with($connection, 'sqlite3://'):
                return new \SupportPal\WhmcsIntegration\Vendor\Symfony\Component\HttpFoundation\Session\Storage\Handler\PdoSessionHandler($connection, $options ?: []);
        }
        throw new \InvalidArgumentException(\sprintf('Unsupported Connection: "%s".', $connection));
    }
}
